import gevent


def force_kill_greenlet(g, kill_timeout=1.0, ignore_greenlet_exit=False, log=None):
    """
    Function that tries to stop a greenlet very hard.
    If ignore_greenlet_exit is set, nothing can stop it: it ignores GreenletExit exceptions.

    :type g: gevent.greenlet.Greenlet
    :type kill_timeout: float
    :type ignore_greenlet_exit: bool
    :type log: logging.Logger
    """
    attempt = 1
    messages = []
    while 1:
        try:
            g.kill(timeout=kill_timeout)
            if g.ready():
                break
        except gevent.GreenletExit:
            if log is not None:
                messages.append(
                    ('Received GreenletExit while waiting for %s to die, ignore_greenlet_exit: %s',
                     (g, ignore_greenlet_exit)))
            if ignore_greenlet_exit:
                pass
            else:
                raise
        if log is not None:
            messages.append(
                ('Attempt #%d failed: %s has not died in %d seconds',
                 (attempt, g, kill_timeout)))
        attempt += 1

    if log is not None:
        for msg, args in messages:
            log.warn(msg, *args)

    return attempt


def gevent_idle_iter(iterable_object, idle_period=100, idle_on_start=False, idle_method=None):
    """
    Make gevent.idle pauses during iteration - call idle_method.
    Do not let gevent block your program!

    :param iterable_object: iterable object
    :type iterable_object: collections.Iterable[T]
    :param idle_period: idle frequency
    :param idle_on_start: make idle before first iteration
    :type idle_on_start: bool
    :param idle_method: idle method object
    :rtype: list[T]
    """
    if idle_method is None:
        idle_method = gevent.idle
    for iterable_item_id, iterable_item in enumerate(iterable_object, 0 if idle_on_start else 1):
        if not iterable_item_id % idle_period:
            idle_method()
        yield iterable_item
