# coding: utf-8
import logging
import functools

import gevent

from .util import force_kill_greenlet


class GreenThread(object):
    """
    A simple green runnable.
    Override method "run" and be good.
    """
    def __init__(self):
        self.__main_task = None

    @staticmethod
    def _log_exception(func):
        @functools.wraps(func)
        def wrapper(*args, **kwargs):
            try:
                func(*args, **kwargs)
            except gevent.GreenletExit:
                raise
            except:
                logging.getLogger().exception("the greenlet died because of unhandled exception")
        return wrapper

    def run(self):
        raise NotImplementedError

    def start(self):
        if self.__main_task is None:
            self.__main_task = gevent.spawn(self._log_exception(self.run))

    def is_running(self):
        return bool(self.__main_task)

    def get(self, timeout=None):
        if self.__main_task is None:
            raise RuntimeError("green thread is not running")
        return self.__main_task.get(timeout=timeout)

    def wait(self, timeout=None):
        return self.get(timeout=timeout)

    def stop(self, ignore_greenlet_exit=False, log=None, kill_timeout=None):
        if self.__main_task is None:
            return 0
        attempts = force_kill_greenlet(self.__main_task,
                                       ignore_greenlet_exit=ignore_greenlet_exit,
                                       log=log,
                                       kill_timeout=kill_timeout)
        self.__main_task = None
        return attempts

    def ready(self):
        return self.__main_task.ready()
