# coding: utf-8
from kazoo.exceptions import NoNodeError, NodeExistsError
from kazoo.retry import ForceRetryError, KazooRetry


def member_id_to_node_name(member_id):
    return member_id


def node_name_to_member_id(node_name):
    return node_name


class SingletonParty(object):
    def __init__(self, client, path, identifier=None):
        """
        A modification of the standard kazoo's party.
        Does not include uuids its child nodes, persistently waits
        until node from previous kazoo session disappears.

        :type client: kazoo.client.KazooClient
        :param path: The party path to use.
        :param identifier: An identifier to use for this member of the
                           party when participating.
        """
        self.client = client
        self.path = path
        self.ensured_path = False
        self.participating = False

        self._retry = KazooRetry(max_tries=-1, max_delay=20)
        self.create_path = self.path + "/" + identifier

    def _ensure_parent(self):
        if not self.ensured_path:
            # make sure our parent node exists
            self.client.ensure_path(self.path)
            self.ensured_path = True

    def join(self):
        """Join the party"""
        return self._retry(self._inner_join)

    def _inner_join(self):
        self._ensure_parent()
        try:
            self.client.create(self.create_path, ephemeral=True)
        except NodeExistsError:
            try:
                _, stat = self.client.get(self.create_path)
            except NoNodeError:
                raise ForceRetryError()
            else:
                client_session_id, _ = self.client.client_id
                if stat.owner_session_id == client_session_id:
                    self.participating = True
                else:
                    raise ForceRetryError()
        else:
            self.participating = True

    def leave(self):
        """Leave the party"""
        self.participating = False
        return self._retry(self._inner_leave)

    def _inner_leave(self):
        try:
            self.client.delete(self.create_path)
        except NoNodeError:
            return False
        return True

    def __len__(self):
        """Return a count of participating clients"""
        self._ensure_parent()
        return len(self._get_children())

    def _get_children(self):
        return self._retry(self.client.get_children, self.path)

    def __iter__(self):
        """Get a list of participating clients' identifiers"""
        self._ensure_parent()
        children = self._get_children()
        for child in children:
            yield node_name_to_member_id(child)

    def list_member_ids(self):
        rv = []
        seen_member_ids = set()
        for member_id in self:
            if member_id not in seen_member_ids:
                rv.append(member_id)
                seen_member_ids.add(member_id)
        return rv
