# coding: utf-8
import gevent


def force_kill_greenlet(g, kill_timeout=None, ignore_greenlet_exit=False, log=None):
    """
    Function that tries to stop a greenlet very hard.
    If ignore_greenlet_exit is set, nothing can stop it: it ignores GreenletExit exceptions.

    :type g: gevent.greenlet.Greenlet
    :type kill_timeout: float
    :type ignore_greenlet_exit: bool
    :type log: logging.Logger
    """
    if kill_timeout is None:
        kill_timeout = 1.0
    attempt = 1
    messages = []
    while 1:
        try:
            g.kill(timeout=kill_timeout)
            if g.ready():
                break
        except gevent.GreenletExit:
            if log is not None:
                messages.append(
                    ('Received GreenletExit while waiting for %s to die, ignore_greenlet_exit: %s',
                     (g, ignore_greenlet_exit)))
            if ignore_greenlet_exit:
                pass
            else:
                raise
        if log is not None:
            messages.append(
                ('Attempt #%d failed: %s has not died in %d seconds',
                 (attempt, g, kill_timeout)))
        attempt += 1

    if log is not None:
        for msg, args in messages:
            log.warn(msg, *args)

    return attempt


class Spawner(object):
    """
    Context manager that spawns greenlets by delegating it to gevent and
    kills all spawned greenlets on exit.
    It is useful for cleaning up all spwaned greenlets when execution aborted unexpectedly.
    Be careful using it: all spawned greenlets will be killed on exit,
    so do exit only when you sure all greenlets are done.
    """
    def __init__(self, kill_timeout, log=None):
        self._gs = []
        self.kill_timeout = kill_timeout
        self.log = log

    def spawn(self, function, *args, **kwargs):
        g = gevent.spawn(function, *args, **kwargs)
        self._gs.append(g)
        return g

    def __enter__(self):
        return self

    def __exit__(self, *args):
        try:
            gevent.killall(self._gs, timeout=self.kill_timeout)
        except gevent.Timeout:
            if self.log is not None:
                # Just ignore gevent.Timeout exception on cleaning up
                self.log.warn('Timeout exceeded during cleaning up spawned greenlets. '
                              'Greenlets count: %d', len(self._gs))
