from __future__ import absolute_import
from __future__ import unicode_literals

import gevent
import requests
import six
from requests.adapters import HTTPAdapter
from infra.swatlib.metrics import InstrumentedSession
from six.moves.urllib import parse as urlparse

DEFAULT_REQ_TIMEOUT = 60


def http_error_parser(e):
    """
    :type e: requests.HTTPError
    :rtype: str
    """
    return 'Request "{}" failed with {} {}: "{}"'.format(e.request.url,
                                                         e.response.status_code,
                                                         e.response.reason,
                                                         e.response.content)


class HttpClientException(Exception):
    def __init__(self, *args, **kwargs):
        self.exc = kwargs.pop('exc', None)
        self.resp = kwargs.pop('resp', None)
        super(HttpClientException, self).__init__(*args, **kwargs)


class HttpClient(object):
    @classmethod
    def from_config(cls, d):
        return cls(**d)

    def __init__(self, client_name, exc_cls, base_url=None, req_timeout=None, max_retries=None,
                 token=None, verify=True, error_parser=None, connection_timeout=None, headers=None):
        """
        :param max_retries:
            From requests docs:
            The maximum number of retries each connection
            should attempt. Note, this applies only to failed DNS lookups, socket
            connections and connection timeouts, never to requests where data has
            made it to the server.
        """
        super(HttpClient, self).__init__()
        self._exc_cls = exc_cls
        self._base_url = base_url
        self._req_timeout = req_timeout
        self._token = token
        self._error_parser = error_parser
        self._session = self._create_session(client_name=client_name, oauth_token=token,
                                             verify=verify, max_retries=max_retries,
                                             connection_timeout=connection_timeout,
                                             headers=headers)

    @staticmethod
    def _create_session(client_name, oauth_token, verify, max_retries=None, connection_timeout=None, headers=None):
        s = InstrumentedSession(client_name, connection_timeout=connection_timeout)
        if oauth_token is not None:
            s.headers['Authorization'] = 'OAuth {}'.format(oauth_token)
        if headers:
            s.headers.update(headers)
        s.verify = verify
        if max_retries is not None:
            s.mount('https://', HTTPAdapter(max_retries=max_retries))
            s.mount('http://', HTTPAdapter(max_retries=max_retries))
        return s

    def _parse_message_from_http_error(self, e):
        """
        :type e: requests.HTTPError
        :rtype: str
        """
        if self._error_parser:
            return self._error_parser(e)
        return six.text_type(e)

    def request(self, method, rel_url, return_json=True, request_timeout=None, **kwargs):
        full_url = urlparse.urljoin(self._base_url, rel_url)
        resp = None
        try:
            with gevent.Timeout(request_timeout or self._req_timeout):
                try:
                    resp = self._session.request(method, full_url, **kwargs)
                    resp.raise_for_status()
                except requests.HTTPError as e:
                    message = self._parse_message_from_http_error(e)
                    raise self._exc_cls(message, resp=resp, exc=e)
                except requests.RequestException as e:
                    raise self._exc_cls(six.text_type(e), resp=resp, exc=e)
        except gevent.Timeout as e:
            raise self._exc_cls('Request timeout exceeded', resp=resp, exc=e)
        if return_json:
            try:
                return resp.json()
            except ValueError as e:
                raise self._exc_cls('Server returned an invalid JSON response', resp=resp, exc=e)
        else:
            return resp

    def get(self, rel_url, request_timeout=None, **kwargs):
        return self.request('GET', rel_url, request_timeout=request_timeout, **kwargs)

    def post(self, rel_url, request_timeout=None, **kwargs):
        return self.request('POST', rel_url, request_timeout=request_timeout, **kwargs)

    def put(self, rel_url, request_timeout=None, **kwargs):
        return self.request('PUT', rel_url, request_timeout=request_timeout, **kwargs)

    def delete(self, rel_url, request_timeout=None, **kwargs):
        return self.request('DELETE', rel_url, request_timeout=request_timeout, **kwargs)

    def stop(self):
        self._session.close()
