# coding: utf-8
"""
Http-GridFS (https://bb.yandex-team.ru/projects/NANNY/repos/http-gridfs/) client.
"""
import collections
import inject
from sepelib.http.request import json_request
from sepelib.yandex import ApiRequestException
from six.moves import http_client as httplib

from infra.swatlib.metrics import InstrumentedSession


class HttpGridfsApiRequestException(ApiRequestException):
    """Common-case exception for http-gridfs service request"""


HttpGridfsFile = collections.namedtuple('HttpGridfsFile', ['id', 'url', 'checksum_url'])


class IHttpGridfsClient(object):
    """
    Interface to be used in dependency injection.
    """

    @classmethod
    def instance(cls):
        """
        :rtype: HttpGridfsClient
        """
        return inject.instance(cls)


class HttpGridfsClient(IHttpGridfsClient):
    _DEFAULT_BASE_URL = 'http://http-gridfs.yandex-team.ru/'
    _DEFAULT_REQ_TIMEOUT = 30
    _DEFAULT_ATTEMPTS = 3

    @classmethod
    def from_config(cls, d):
        return cls(**d)

    def __init__(self, url=None, req_timeout=None, attempts=None):
        self._base_url = (url or self._DEFAULT_BASE_URL).rstrip('/')
        self._req_timeout = req_timeout or self._DEFAULT_REQ_TIMEOUT
        self._attempts = attempts or self._DEFAULT_ATTEMPTS
        self._session = InstrumentedSession('httpgridfs')

    def put(self, content, content_type=None, custom_url=None):
        """
        Put file to GridFS

        :type content: six.text_type
        :type content_type: six.text_type
        :type custom_url: six.text_type
        :rtype: HttpGridfsFile
        """
        if custom_url:
            base_url = custom_url.rstrip('/')
            if base_url.endswith('/v1'):
                base_url = base_url[:-3]
        else:
            base_url = self._base_url
        url = '{base_url}/v1/resources/'.format(base_url=base_url)
        response = json_request(
            'post', url,
            ok_statuses=[httplib.OK, httplib.CREATED],
            timeout=self._req_timeout,
            exception=HttpGridfsApiRequestException,
            session=self._session,
            data=content,
            headers={'Content-Type': content_type}
        )
        id_ = response['id']
        checksum = response['checksum']
        return HttpGridfsFile(
            id=id_,
            url='{base_url}/v1/resources/{id}/'.format(base_url=self._base_url, id=id_),
            checksum_url='{base_url}/v2/resources/{checksum}/'.format(base_url=self._base_url, checksum=checksum)
        )
