import requests
import inject
from infra.swatlib import httpclient


class InfraError(httpclient.HttpClientException):
    pass


class IInfraClient(object):
    @classmethod
    def instance(cls):
        return inject.instance(cls)

    def get_service(self, s_id):
        raise NotImplementedError

    def get_environment(self, e_id):
        raise NotImplementedError

    def get_admins(self, s_id):
        raise NotImplementedError

    def get_events(self, events_filter):
        raise NotImplementedError

    def create_event(self, event_param):
        raise NotImplementedError

    def update_event(self, event_id, event_param):
        raise NotImplementedError


class InfraClient(IInfraClient):
    INFRA_API_URL = 'https://infra-api.yandex-team.ru/'
    DEFAULT_REQ_TIMEOUT = 10  # seconds
    DEFAULT_VERIFY_SSL = True

    @classmethod
    def from_config(cls, d):
        return cls(url=d.get('api_url'),
                   token=d.get('oauth_token'),
                   req_timeout=d.get('req_timeout'),
                   verify_ssl=d.get('verify_ssl'),
                   max_retries=d.get('max_retries'))

    def __init__(self, url=None, token=None, req_timeout=None, verify_ssl=None,
                 max_retries=None):
        self.url = url or self.INFRA_API_URL
        req_timeout = req_timeout or self.DEFAULT_REQ_TIMEOUT
        token = token or None
        verify_ssl = self.DEFAULT_VERIFY_SSL if verify_ssl is None else verify_ssl
        self._client = httpclient.HttpClient(
            client_name='infra',
            exc_cls=InfraError,
            base_url=self.url,
            req_timeout=req_timeout,
            token=token,
            verify=verify_ssl,
            max_retries=max_retries,
            error_parser=httpclient.http_error_parser,
        )
        self.session = requests.Session()

    def get_environment(self, e_id):
        rel_url = 'v1/environments/{}'.format(e_id)
        return self._client.get(rel_url)

    def get_service(self, s_id):
        rel_url = 'v1/services/{}'.format(s_id)
        return self._client.get(rel_url)

    def get_admins(self, s_id):
        rel_url = 'v1/acl?serviceId={}'.format(s_id)
        return self._client.get(rel_url)

    def get_events(self, events_filter):
        rel_url = 'v1/events'
        return self._client.get(rel_url, params=events_filter)

    def create_event(self, event_param):
        rel_url = 'v1/events'
        return self._client.post(rel_url, json=event_param)

    def update_event(self, event_id, event_param):
        rel_url = 'v1/events/{}'.format(event_id)
        return self._client.put(rel_url, json=event_param)
