from __future__ import unicode_literals

from infra.swatlib import pbutil


class Codec(object):
    def decode(self, d):
        """
        Decodes k8s dict into object.

        :type d: dict
        :rtype: objects
        """
        raise NotImplementedError

    def encode(self, obj):
        """
        Encodes provided object into k8s dict.

        :type obj: object
        :rtype: dict
        """
        raise NotImplementedError

    def set_meta(self, obj, d):
        """
        Translates k8s metadata and sets it to object's meta.

        :type obj: object
        :type d: dict
        """
        raise NotImplementedError


class MetadataPbConverter(object):
    """
    Kube's metadata is not supported by CRD. So we need custom converter
    to convert dict to/from pb.
    """
    def kubedict_to_pb(self, d, meta_pb):
        raise NotImplementedError

    def pb_to_kubedict(self, meta_pb):
        raise NotImplementedError


class PbCodec(Codec):
    def __init__(self, pb_cls, group, api_version, kind, metadata_pb_converter):
        self.pb_cls = pb_cls
        self.api_version = '{}/{}'.format(group, api_version)
        self.kind = kind
        self.metadata_pb_converter = metadata_pb_converter

    def set_meta(self, obj, d):
        self.metadata_pb_converter.kubedict_to_pb(d['metadata'], obj.meta)

    def encode(self, obj):
        return {
            'apiVersion': self.api_version,
            'kind': self.kind,
            'metadata': self.metadata_pb_converter.pb_to_kubedict(obj.meta),
            'spec': pbutil.pb_to_jsondict(obj.spec)
        }

    def decode(self, d):
        obj = self.pb_cls()
        self.set_meta(obj, d)
        pbutil.jsondict_to_pb(d['spec'], obj.spec)
        return obj
