# coding: utf-8
import logging

import inject
import enum
from six.moves import http_client as httplib
from sepelib.core import config as appconfig
# these imports come from the orly_client package
from infra.orly import client
from infra.orly.proto import orly_pb2


class IOrlyClient(object):
    """
    Interface to be used in dependency injection.
    """

    @classmethod
    def instance(cls):
        """
        :rtype: OrlyClient
        """
        return inject.instance(cls)


class OperationOutcome(enum.Enum):
    ALLOWED = 0
    FORBIDDEN = 1
    UNAVAILABLE = 2


class OrlyClient(IOrlyClient):
    DEFAULT_ORLY_URL = u'https://orly.nanny.yandex-team.ru'

    def __init__(self, url=DEFAULT_ORLY_URL):
        """
        :type url: six.text_type
        """
        self._client = client.Orly(url=url)

    @classmethod
    def from_config(cls, config):
        return cls(url=config['url'])

    def start_operation(self, rule, id_, labels=None):
        """
        :type rule: six.text_type
        :type id_: six.text_type
        :type labels: list[(six.text_type, six.text_type)] | None
        :rtype: (OperationStatus.*, six.text_type)
        """
        req_pb = orly_pb2.OperationRequest(rule=rule, id=id_)
        if labels:
            req_pb.labels.update(labels)
        resp_pb, status_pb = self._client.start_operation(req_pb)
        if status_pb is None:
            return OperationOutcome.ALLOWED, u'successfully started operation {}'.format(resp_pb.operation.meta.id)
        elif status_pb.status == u'Failure' and status_pb.code == httplib.CONFLICT:
            return OperationOutcome.FORBIDDEN, u'failed to start operation: {}'.format(status_pb.message)
        else:
            return OperationOutcome.UNAVAILABLE, u'{} (status: {}, code: {})'.format(
                status_pb.message, status_pb.status, status_pb.code)


class OrlyBrakeApplied(Exception):
    pass


class OrlyBrake(object):
    _client = inject.attr(IOrlyClient)  # type: OrlyClient

    def __init__(self, rule, metrics_registry, force_enabled=False):
        """
        :type rule: six.text_type
        :type metrics_registry: infra.swatlib.metrics.Registry
        :type force_enabled: bool
        """
        self._rule = rule

        registry = metrics_registry.path(u'orly-brake-{}'.format(rule))
        self._allowed_counter = registry.get_counter(u'allowed')
        self._forbidden_counter = registry.get_counter(u'forbidden')
        self._transaction_error_counter = registry.get_counter(u'forbidden-by-transaction-error')
        self._unavailable_counter = registry.get_counter(u'unavailable')
        self._force_enabled = force_enabled

        self._logger = logging.getLogger(u'orly-brake({})'.format(rule))

    def is_enabled(self):
        if self._force_enabled:
            return True
        is_production = appconfig.get_value(u'run.production', default=False)
        if not is_production:
            # to avoid accidentally sharing operations limit between production and testing
            return False
        enabled_rules = set(appconfig.get_value(u'run.enabled_orly_brake_rules', default=[]))
        return self._rule in enabled_rules

    def maybe_apply(self, op_id, op_labels=None, op_log=None):
        """
        :type op_id: six.text_type
        :type op_labels: list[(six.text_type, six.text_type)]
        :type op_log: logging.Logger | logging.LoggerAdapter | None
        :raises: OrlyBrakeApplied
        """
        if not self.is_enabled():
            return

        log = op_log or self._logger

        log.debug(u'Starting orly operation %s, rule "%s"', op_id, self._rule)
        outcome, reason = self._client.start_operation(rule=self._rule, id_=op_id, labels=op_labels)

        # use combination of .format- and %s-formatting to avoid merging them in Sentry:
        if outcome == OperationOutcome.ALLOWED:
            log.debug(u'orly (rule "{}") explicitly allows operation "%s": %s'.format(self._rule), op_id, reason)
        elif outcome == OperationOutcome.FORBIDDEN:
            self._forbidden_counter.inc()
            if u'transaction conflict' in reason:
                # temporary counter to let us know when orly experiences its internal issues
                self._transaction_error_counter.inc()
            log.debug(u'orly (rule "{}") explicitly forbids operation "%s": %s'.format(self._rule), op_id, reason)
            raise OrlyBrakeApplied(reason)
        elif outcome == OperationOutcome.UNAVAILABLE:
            self._unavailable_counter.inc()
            log.warn(u'failed to check whether orly (rule "{}") '
                     u'allows or forbids operation "%s": %s'.format(self._rule), op_id, reason)
            # if orly is unavailable for any reason, we just continue

        self._allowed_counter.inc()
