"""
Utilities to work with random values
"""
from __future__ import division
import random
import struct
from six.moves import range

# Copy paste from base64 module with lower case letters and no padding.
_b32alphabet = {
    0: 'a', 9: 'j', 18: 's', 27: '3',
    1: 'b', 10: 'k', 19: 't', 28: '4',
    2: 'c', 11: 'l', 20: 'u', 29: '5',
    3: 'd', 12: 'm', 21: 'v', 30: '6',
    4: 'e', 13: 'n', 22: 'w', 31: '7',
    5: 'f', 14: 'o', 23: 'x',
    6: 'g', 15: 'p', 24: 'y',
    7: 'h', 16: 'q', 25: 'z',
    8: 'i', 17: 'r', 26: '2',
}

_b32tab = [v for k, v in sorted(_b32alphabet.items())]


def _b32encode(s):
    """Encode a string using Base32.

    s is the string to encode.  The encoded string is returned.
    """
    parts = []
    quanta, _ = divmod(len(s), 5)
    for i in range(quanta):
        # c1 and c2 are 16 bits wide, c3 is 8 bits wide.  The intent of this
        # code is to process the 40 bits in units of 5 bits.  So we take the 1
        # leftover bit of c1 and tack it onto c2.  Then we take the 2 leftover
        # bits of c2 and tack them onto c3.  The shifts and masks are intended
        # to give us values of exactly 5 bits in width.
        c1, c2, c3 = struct.unpack('!HHB', s[i * 5:(i + 1) * 5])
        c2 += (c1 & 1) << 16  # 17 bits wide
        c3 += (c2 & 3) << 8  # 10 bits wide
        parts.extend([_b32tab[c1 >> 11],  # bits 1 - 5
                      _b32tab[(c1 >> 6) & 0x1f],  # bits 6 - 10
                      _b32tab[(c1 >> 1) & 0x1f],  # bits 11 - 15
                      _b32tab[c2 >> 12],  # bits 16 - 20 (1 - 5)
                      _b32tab[(c2 >> 7) & 0x1f],  # bits 21 - 25 (6 - 10)
                      _b32tab[(c2 >> 2) & 0x1f],  # bits 26 - 30 (11 - 15)
                      _b32tab[c3 >> 5],  # bits 31 - 35 (1 - 5)
                      _b32tab[c3 & 0x1f],  # bits 36 - 40 (1 - 5)
                      ])
    return ''.join(parts)


def gen_random_str(bits=120, src=random):
    """
    Generates random string with :param bits: of entropy.

    :type bits: int
    :type src: random.Random
    :rtype: str
    """
    if bits % 8 != 0:
        raise ValueError('bits must be multiple of 8, got {}'.format(bits))
    b = bytearray(src.getrandbits(8) for _ in range(bits // 8))
    return _b32encode(b)


if __name__ == '__main__':
    print(gen_random_str())
    print(gen_random_str(64))
    print(gen_random_str(128))
    print(gen_random_str(src=random.SystemRandom()))
