from __future__ import unicode_literals

import cachetools
import inject

from sepelib.core import config
from sepelib.yandex import oauth

from infra.swatlib.auth import passport
from infra.swatlib.auth import tvm
from . import exceptions


class IRpcAuthenticator(object):
    def authenticate_request(self, flask_request):
        """
        Authenticates provided user request.

        :param flask_request: Flask request object.

        :rtype: AuthSubject
        """
        raise NotImplementedError


class Request(object):
    """
    Http request which need to be authenticated.

    Usually constructed from flask.Request, but for ease of testing,
    we use this one, because flask.Request requires wsgi environment.
    """
    __slots__ = ['url', 'host', 'user_ip', 'auth_header', 'session_id', 'tvm_ticket']

    def __init__(self, url, host, user_ip, auth_header, session_id, tvm_ticket):
        """
        :param url: Request URL
        :param host: Host header value
        :param user_ip: User IP (from X-Forwarded-For or peer address)
        :param auth_header: Authorization header content
        :param session_id: Session_id cookie value
        :param tvm_ticket: TVM ticket header content
        """
        self.url = url
        self.host = host
        self.user_ip = user_ip
        self.auth_header = auth_header
        self.session_id = session_id
        self.tvm_ticket = tvm_ticket


class AuthSubject(object):
    """
    Information about authentication subject.

    Currently we only have login.
    """
    # Use slots, because there will be a lot of these objects.
    # Almost every request will generate one
    __slots__ = ['login', 'tvm_client_id']

    def __init__(self, login, tvm_client_id=None):
        self.login = login
        self.tvm_client_id = tvm_client_id


USE_INJECTED = object()


class CachingPassportAuthenticator(IRpcAuthenticator):
    """
    Performs authentication via passport.

    :type oauth_client: oauth.IOAuth
    :type passport_client: passport.IPassportClient
    """
    ANON_RESULT = AuthSubject('anonymous')
    CACHE_SIZE = 1000
    # This should also work - don't think that using invalid cookie for some time is a crime.
    # But this will allow us not to issue blackbox requests on subsequent ajax requests to nanny.
    # This will help in case blackbox and/or staff API blackouts. There have been incidents already.
    # See https://st.yandex-team.ru/SWAT-1762 for details.
    CACHE_TTL = 60 * 60 * 24

    oauth_client = inject.attr(oauth.IOAuth)
    passport_client = inject.attr(passport.IPassportClient)
    tvm_client = inject.attr(tvm.ITvmClient)

    def __init__(self, oauth_client, passport_client, is_auth_disabled, tvm_client=None, force_return_user=None):
        """
        :type oauth_client: oauth.IOAuth | USE_INJECTED
        :type passport_client: passport.IPassportClient | USE_INJECTED
        :type is_auth_disabled: bool
        :type tvm_client: tvm.ITvmClient | None | USE_INJECTED
        """
        if oauth_client is not USE_INJECTED:
            self.oauth_client = oauth_client
        if passport_client is not USE_INJECTED:
            self.passport_client = passport_client
        if tvm_client is not USE_INJECTED:
            self.tvm_client = tvm_client
        self.is_auth_disabled = is_auth_disabled
        self.force_return_user = AuthSubject(force_return_user) if force_return_user else None
        self.session_id_cache = cachetools.TTLCache(maxsize=self.CACHE_SIZE, ttl=self.CACHE_TTL)
        self.oauth_header_cache = cachetools.TTLCache(maxsize=self.CACHE_SIZE, ttl=self.CACHE_TTL)

    @classmethod
    def from_inject(cls, oauth_client=USE_INJECTED, passport_client=USE_INJECTED, tvm_client=USE_INJECTED):
        return cls(
            oauth_client=oauth_client,
            passport_client=passport_client,
            tvm_client=tvm_client,
            is_auth_disabled=False)

    def authenticate_via_oauth(self, request):
        if not request.auth_header:
            return None
        login = self.oauth_header_cache.get(request.auth_header)
        if login is not None:
            return AuthSubject(login)
        try:
            login = self.oauth_client.get_user_login_by_authorization_header(
                self.passport_client,
                request.auth_header,
                request.user_ip)
        except Exception as e:
            raise exceptions.UnauthenticatedError('Failed to authenticate: {}'.format(e))
        self.oauth_header_cache[request.auth_header] = login
        return AuthSubject(login)

    def authenticate_via_session_cookie(self, request):
        if not request.session_id:
            return None
        login = self.session_id_cache.get(request.session_id)
        if login is not None:
            return AuthSubject(login)
        try:
            passport_result = self.passport_client.check_passport_cookie(
                cookies={'Session_id': request.session_id},
                host=request.host,
                user_ip=request.user_ip,
                request_url=request.url
            )
        except Exception as e:
            raise exceptions.UnauthenticatedError('Failed to authenticate: {}'.format(e))
        if passport_result.redirect_url:
            raise exceptions.UnauthenticatedError('Cookie needs updating',
                                                  redirect_url=passport_result.redirect_url)
        self.session_id_cache[request.session_id] = passport_result.login
        return AuthSubject(passport_result.login)

    def authenticate_via_tvm_ticket(self, request):
        if not request.tvm_ticket:
            return None
        try:
            tvm_result = self.tvm_client.check_service_ticket(request.tvm_ticket)
        except Exception as e:
            raise exceptions.UnauthenticatedError('Failed to authenticate: {}'.format(e))
        return AuthSubject(login='', tvm_client_id=tvm_result.src)

    def authenticate_request(self, request):
        """
        Attempts to authenticate request. Returns AuthSubject object on success.
        Otherwise raises RPC exception error.

        :type request: Request
        """
        # Respect configuration parameters.
        if self.is_auth_disabled or not config.get_value('run.auth'):
            return self.force_return_user or self.ANON_RESULT
        auth_subject = self.authenticate_via_oauth(request)
        if auth_subject is not None:
            return auth_subject
        auth_subject = self.authenticate_via_session_cookie(request)
        if auth_subject is not None:
            return auth_subject
        if self.tvm_client is not None:
            auth_subject = self.authenticate_via_tvm_ticket(request)
            if auth_subject is not None:
                return auth_subject
        raise exceptions.UnauthenticatedError("Request doesn't have proper authentication credentials")
