"""
Exceptions that can be raised by rpc handlers and caught by machinery in rpc blueprint.

Exceptions match those errors defined in protobuf.
"""
from __future__ import unicode_literals

from six.moves import http_client as httplib

__all__ = ['RpcError',
           'InternalError', 'BadRequestError', 'UnauthenticatedError',
           'ForbiddenError', 'NotFoundError', 'ConflictError', 'ServiceUnavailable']


class RpcError(Exception):
    def __init__(self, message, redirect_url=None):
        super(RpcError, self).__init__(message)
        self.redirect_url = redirect_url

    status = None  # Http status code for this particular error


class InternalError(RpcError):
    """
    Internal error happened.
    """
    status = httplib.INTERNAL_SERVER_ERROR


class BadRequestError(RpcError):
    """
    Request is not valid.
    """
    status = httplib.BAD_REQUEST


class UnauthenticatedError(RpcError):
    """
    Request doesn't have proper authentication credentials.
    """
    status = httplib.UNAUTHORIZED


class ForbiddenError(RpcError):
    """
    Caller does not have permission to perform this action.
    """
    status = httplib.FORBIDDEN


class NotFoundError(RpcError):
    """
    Requested entity not found.
    """
    status = httplib.NOT_FOUND


class ConflictError(RpcError):
    """
    Action conflicts with current system state (e.g. compare-and-swap failed)
    """
    def __init__(self, message, redirect_url=None, conflicts=None):
        super(ConflictError, self).__init__(message, redirect_url)
        self.conflicts = conflicts or []

    status = httplib.CONFLICT


class ServiceUnavailable(RpcError):
    """
    Internal error happened.
    """
    status = httplib.SERVICE_UNAVAILABLE
