from __future__ import absolute_import

import flask
import gevent
import requests
from sepelib.flask.auth.util import login_exempt


class Behavior(object):
    simulate_memory_leak = 0
    simulate_cpu_load = 0
    stop_web_server = 0
    sleep_seconds = 0
    return_status_code = 200

    def __init__(self, flask_request, can_load_behavior_from=False):
        load_behavior_from = flask_request.args.get('load_behavior_from')
        if can_load_behavior_from and load_behavior_from:
            url = "{}?request_id={}".format(load_behavior_from, flask_request.headers.get('X-Req-Id'))
            resp = requests.get(url)
            resp.raise_for_status()
            source = resp.json()
        else:
            source = flask_request.args

        self.simulate_cpu_load = int(source.get('simulate_cpu_load', self.simulate_cpu_load))
        self.simulate_memory_leak = int(source.get('simulate_memory_leak', self.simulate_memory_leak))
        self.stop_web_server = int(source.get('stop_web_server', self.stop_web_server))
        self.sleep_seconds = float(source.get('sleep_seconds', self.sleep_seconds))
        self.return_status_code = int(source.get('return_status_code', self.return_status_code))


class SimulateBehaviorExt(object):

    def __init__(self, web_server, config, app=None):
        self.app = None
        self.web_server = web_server
        self.enabled = config.get('enabled')
        self.url = config.get('url', '/_simulate_behavior')
        self.can_load_behavior_from = config.get('can_load_behavior_from')
        self.can_simulate_memory_leak = config.get('can_simulate_memory_leak')
        self.can_simulate_cpu_load = config.get('can_simulate_cpu_load')
        self.can_stop_web_server = config.get('can_stop_web_server')
        if app:
            self.init_app(app)

    def init_app(self, app):
        self.app = app
        if self.enabled:
            self.app.add_url_rule(self.url, view_func=self.backend_behavior, methods=['GET', 'POST', 'PUT'])

    @login_exempt
    def backend_behavior(self):
        behavior = Behavior(flask.request, can_load_behavior_from=self.can_load_behavior_from)

        if behavior.sleep_seconds:
            gevent.sleep(behavior.sleep_seconds)

        if self.can_simulate_cpu_load and behavior.simulate_cpu_load:
            self._simulate_cpu_load(behavior.simulate_cpu_load)

        if self.can_simulate_memory_leak and behavior.simulate_memory_leak:
            self._simulate_memory_leak(behavior.simulate_memory_leak)

        if self.can_stop_web_server and behavior.stop_web_server:
            self.web_server.stop()

        return flask.Response('ok', status=behavior.return_status_code)

    @staticmethod
    def _simulate_cpu_load(seconds):
        timeout = gevent.Timeout(seconds, ValueError('stop'))
        timeout.start()
        x = 23493939
        try:
            i = 0
            while True:
                try:
                    i += 1
                    x * x
                    if i % 10000:
                        gevent.sleep()
                except ValueError:
                    break
        finally:
            timeout.close()

    @staticmethod
    def _simulate_memory_leak(seconds):
        timeout = gevent.Timeout(seconds, ValueError('stop'))
        timeout.start()
        i, ar = 0, []
        try:
            while True:
                try:
                    i += 1
                    ar.append(' ' * 10 ** 6)
                    if i % 100:
                        gevent.sleep()
                except ValueError:
                    break
        finally:
            timeout.close()
            del ar
