"""
Thrift-to-json encoder. Borrowed from twitter:
https://github.com/twitter/commons/blob/master/src/python/twitter/thrift/text/thrift_json_encoder.py
"""
import json

import enum
from thrift.Thrift import TType
import six

if six.PY2:
    long_int = long  # noqa
else:
    long_int = int


class ThriftJSONEncoder(json.JSONEncoder):
    """An encoder that makes python thrift structs JSON serializable via the
    standard python json module.

    Pass this encoder when writing json, like this:

    json.dumps(thriftobj, cls=text.ThriftJSONEncoder, <other kwargs>)

    Note that this is not a full protocol implementation in the thrift sense. This
    is just a quick-and-easy pretty-printer for unittests, debugging etc.
    """
    THRIFT_SPEC = 'thrift_spec'

    def __init__(self, *args, **kwargs):
        super(ThriftJSONEncoder, self).__init__(*args, **kwargs)

    def default(self, o):
        # handle sets (the value type of a thrift set field). We emit them as lists,
        # sorted by element.
        if isinstance(o, set):
            ret = list(o)
            ret.sort()
            return ret

        # handle everything that isn't a thrift struct.
        if not hasattr(o, ThriftJSONEncoder.THRIFT_SPEC):
            return super(ThriftJSONEncoder, self).default(o)

        # handle thrift structs.
        spec = getattr(o, ThriftJSONEncoder.THRIFT_SPEC)
        ret = {}
        for (field_number, type_, name, type_info, default) in \
                [field_spec for field_spec in spec if field_spec is not None]:
            if name in o.__dict__:
                val = o.__dict__[name]
                if isinstance(val, enum.IntEnum):
                    val = val.name
                ret[name] = val
        return ret


def thrift_to_json(o, sort_keys=False, **kwargs):
    """
    A utility shortcut function to return a pretty-printed JSON thrift object.
    """
    return json.dumps(o, sort_keys=sort_keys, cls=ThriftJSONEncoder, **kwargs)


def encode_thrift_to_dict(obj):
    """
    Converts thrift object :param obj: to dict

    :rtype: dict
    """
    ret = {}
    spec_list = [field_spec for field_spec in obj.thrift_spec if field_spec is not None]
    for (field_number, ttype, name, type_info, default) in spec_list:
        val = getattr(obj, name)
        if val is None:
            ret[name] = None
        elif ttype == TType.STRUCT:
            ret[name] = encode_thrift_to_dict(val)
        elif ttype == TType.LIST:
            ret[name] = [encode_thrift_to_dict(i) for i in val]
        elif ttype == TType.SET:
            ret[name] = {encode_thrift_to_dict(i) for i in val}
        elif ttype == TType.MAP:
            ret[name] = {encode_thrift_to_dict(k): encode_thrift_to_dict(v) for k, v in six.iteritems(val)}
        elif ttype == TType.STRING:
            ret[name] = six.text_type(val)
        elif ttype == TType.DOUBLE:
            ret[name] = float(val)
        elif ttype == TType.I64:
            ret[name] = long_int(val)
        elif ttype in (TType.I32, TType.I16, TType.BYTE):
            ret[name] = int(val)
        elif ttype == TType.BOOL:
            ret[name] = not not val
        else:
            raise Exception('Unrecognized thrift field type: %d' % ttype)
    return ret
