# coding=utf-8
from __future__ import unicode_literals
import logging
import time
import zlib
import collections
from six.moves.builtins import cmp

import gevent
import requests
import inject

from iss_thrift3 import ttypes as iss_types
from iss_thrift3 import IssService

from infra.swatlib import metrics
from infra.swatlib.thrift.client import ThriftBinClient
import six


class IssApiCallError(Exception):
    """
    Ошибка при вызове api метода звездолёта
    """
    pass


class IssValidationError(ValueError):
    """
    Исключение, вызываемое при неправильных параметрах при работе со звездолётом
    """
    pass


class IssHostConfiguration(object):
    """
    Конфигурация хоста
    """
    def __init__(self, instances, footprint):
        """
        :param list[IssInstance] instances: инстансы
        :param str|None footprint: хеш конфигурации
        """
        self.instances = instances
        self.footprint = footprint


class IssConfiguration(object):
    """
    Описание ISS конфигурации
    """

    def __init__(self, configuration_id, parent=None, created=None, is_applicable=False, properties=None):
        self.id = configuration_id
        self.parent = parent
        self.created = created
        self.applicable = is_applicable
        self.properties = properties

    def is_applicable(self):
        return self.applicable

    @property
    def family(self):
        return IssApiWrapper.get_family_from_configuration_id(self.id)

    @property
    def name(self):
        return IssApiWrapper.get_configuration_name_from_id(self.id)


class IssCacherVersion(object):
    """
    Описание версии ISS кешера
    """

    def __init__(self, version, timestamp, host):
        self.version = version
        self.timestamp = timestamp
        self.host = host

    def __str__(self):
        return 'Version: {}, timestamp: {}, host: {}'.format(self.version, self.timestamp, self.host)

    def __repr__(self):
        return 'IssCacherVersion({}, {}, {})'.format(self.version, self.timestamp, self.host)


class IssFeatures(object):
    """
    Фичи ISS (согласно https://st.yandex-team.ru/SWAT-1892#1439825596000)
    """
    # отключает метод apply, в который переданы целевые состояния с помощью фильтров (закрытие на запись)
    APPLY_BY_FILTERS = 'applyByFilters'
    # отключает метод apply, в который переданы целевые состояния с помощью перечисления инстансов
    APPLY_BY_SLOTS = 'applyBySlots'
    # отключить получение хостовых конфигураций (закрытие на чтение)
    GET_HOST_CONFIGURATION = 'getHostConfiguration'
    # отключение записи текущего состояния
    SAVE_CURRENT_STATE = 'saveCurrentState'
    # отключение ускоренной доставки состояния на хосты
    PUSH = 'push'


class IssStates(object):
    """
    Фиксированные состояния инстанса
    """
    # инстанс готов к запуску
    INSTANCE_PREPARED = 'PREPARED'
    # инстанс запущен
    INSTANCE_ACTIVE = 'ACTIVE'
    # инстанс удалён с машины
    INSTANCE_REMOVED = 'REMOVED'
    # инстанс ещё не трогали
    INSTANCE_UNKNOWN = 'UNKNOWN'


class IssHookTimeLimit(object):
    """
    Объект описания лимитов на хуки ISS
    """
    ISS_CLASS_NAME = 'ru.yandex.iss.TimeLimit'
    ISS_PATH_NAME = 'timeLimits'

    def __init__(self, hook_name, min_restart_period, max_execution_time):
        """
        :param str | unicode hook_name: название хука
        :param int | None min_restart_period: минимальный промежуток времени между запусками хука, в мс
        :param int | None max_execution_time: таймаут на время исполнения хука, в мс
        """
        self.hook_name = hook_name
        self.min_restart_period = min_restart_period
        self.max_execution_time = max_execution_time

    @classmethod
    def get_from_info(cls, hook_name, hook_info):
        min_restart_period = None
        if 'minRestartPeriodMs' in hook_info:
            min_restart_period = int(hook_info['minRestartPeriodMs'])
        max_execution_time = None
        if 'minRestartPeriodMs' in hook_info:
            max_execution_time = int(hook_info['maxExecutionTimeMs'])
        return IssHookTimeLimit(
            hook_name=hook_name,
            min_restart_period=min_restart_period,
            max_execution_time=max_execution_time
        )

    def __repr__(self):
        return 'IssHookTimeLimit("{}", {}, {})'.format(
            self.hook_name, self.min_restart_period, self.max_execution_time
        )


class IssResource(object):
    """
    Объект описания ресурса инстанса в ISS
    """
    ISS_CLASS_NAME = 'ru.yandex.iss.Resource'
    ISS_PATH_NAME = 'resources'

    def __init__(self, name, urls, uuid, size, queue, cached, checksum, check_period):
        """
        :param str | unicode name: локальное имя ресурса
        :param list | tuple | None urls: список источников
        :param int | None size: размер
        :param str | unicode | None queue: очередь закачки
        :param bool | None cached: является ли кешируемым
        :param str | unicode | None checksum: md5 для проверки целостности
        :param int | None check_period: период для проверки целостности
        """
        self.name = name
        self.urls = urls
        self.uuid = uuid
        self.size = size
        self.queue = queue
        self.cached = cached
        self.checksum = checksum
        self.check_period = check_period

    @classmethod
    def get_from_info(cls, hook_name, hook_info):
        min_restart_period = None
        if 'minRestartPeriodMs' in hook_info:
            min_restart_period = int(hook_info['minRestartPeriodMs'])
        max_execution_time = None
        if 'minRestartPeriodMs' in hook_info:
            max_execution_time = int(hook_info['maxExecutionTimeMs'])
        return IssHookTimeLimit(
            hook_name=hook_name,
            min_restart_period=min_restart_period,
            max_execution_time=max_execution_time
        )


class IssShard(object):
    """
    Объект описания шарда инстанса в ISS
    """
    ISS_CLASS_NAME = 'ru.yandex.iss.Shard'

    def __init__(self, shard_name, shard_dir, shard_id, deduplication_mode, queue, cached):
        """
        :param str | unicode shard_name: локальное имя шарда
        :param str | unicode | None shard_dir: директория шарда, если указана
        :param str | unicode | None shard_id: идентификатор шарда
        :param str | unicode | None deduplication_mode: режим дедупликации
        :param str | unicode | None queue: режим дедупликации
        :param bool | None cached: является ли кешируемым
        """
        self.shard_name = shard_name
        self.shard_dir = shard_dir
        self.shard_id = shard_id
        self.deduplication_mode = deduplication_mode
        self.queue = queue
        self.cached = cached

    @classmethod
    def get_from_info(cls, shard_name, shard_info):
        cached = shard_info.get('cached', None)
        if cached == 'false':
            cached = False
        if cached == 'true':
            cached = True
        return IssShard(
            shard_name=shard_name,
            shard_dir=shard_info.get('shardDir'),
            shard_id=shard_info.get('shardId'),
            deduplication_mode=shard_info.get('deduplicationMode'),
            queue=shard_info.get('queue', ''),
            cached=cached
        )


class IssInstance(IssStates):
    """
    Описание инстанса ISS.
    service в мире bsconfig называется port, то есть является портом, на котором поднимается инстанс.
    Для мира ISS это понятие шире, является просто строкой, определяющей инстанс для хоста.
    """

    def __init__(self, configuration_id, host, service, target_state=None, current_state=None, properties=None):
        """
        :param configuration_id: идентификатор конфигурации
        :param host: название хоста
        :param service: название сервиса
        :param target_state: целевое состояние инстанса
        :param current_state: текущее состояние инстанса
        """
        self.configuration_id = configuration_id
        self.host = host
        self.service = service
        self._target_state = None
        self._current_state = None
        if target_state:
            self.target_state = target_state
        if current_state:
            self.current_state = current_state
        self.properties = properties or {}
        self._time_limits = None
        self._resources = None
        self._shards = None

    def _parse_property_info(self):
        """
        Получаем из properties информацию о ресурсах, шардах и таймлимитах
        """
        properties_info = collections.defaultdict(dict)

        # плохо, что два раза проходимся по properties, но так читаемей и
        # не приходится создавать классы со всеми дефолтными полями - выходит надёжней
        for key, value in six.iteritems(self.properties):
            parsed_info = key.split('/', 2)
            if len(parsed_info) == 3:
                _, parsed_name, parsed_property = parsed_info
                properties_info[parsed_name][parsed_property] = value

        self._time_limits = {}
        self._resources = {}
        self._shards = {}

        for property_name, property_info in six.iteritems(properties_info):
            property_class = property_info.get('@class')
            if property_class == IssHookTimeLimit.ISS_CLASS_NAME:
                self._time_limits[property_name] = IssHookTimeLimit.get_from_info(property_name, property_info)
            if property_class == IssShard.ISS_CLASS_NAME:
                self._shards[property_name] = IssShard.get_from_info(property_name, property_info)
            if property_class == IssResource.ISS_CLASS_NAME:
                self._resources[property_name] = IssResource.get_from_info(property_name, property_info)

    @property
    def time_limits(self):
        """
        Лимиты хуков инстанса

        :rtype: dict of IssHookTimeLimit
        """
        if self._time_limits is None:
            self._parse_property_info()
        return self._time_limits

    @property
    def resources(self):
        """
        Ресурсы инстанса

        :rtype: dict of IssResource
        """
        if self._resources is None:
            self._parse_property_info()
        return self._resources

    @property
    def shards(self):
        """
        Словарь с шардами инстанса

        :type: dict of IssShard
        """
        if self._shards is None:
            self._parse_property_info()
        return self._shards

    @property
    def cause(self):
        return self.properties.get('cause')

    @property
    def target_state(self):
        return self._target_state

    @target_state.setter
    def target_state(self, value):
        if value not in (self.INSTANCE_REMOVED, self.INSTANCE_ACTIVE, self.INSTANCE_PREPARED):
            raise IssValidationError('Incorrect target state {} for instance {}/{}'.format(
                value, self.configuration_id, self.slot
            ))
        self._target_state = value

    @property
    def current_state(self):
        return self._current_state

    @current_state.setter
    def current_state(self, value):
        # не можем проверить,  потому как может быть любое значение
        self._current_state = value

    @property
    def slot(self):
        return IssApiWrapper.get_slot_name_from_host_and_service(self.host, self.service)

    def __str__(self):
        return 'Instance of configuration {} on slot {}'.format(self.configuration_id, self.slot)

    def __repr__(self):
        return 'IssInstance({}, {}, {}, {}, {})'.format(
            self.configuration_id, self.host, self.service, self.target_state, self.current_state
        )

    def is_active(self):
        return self.current_state == self.INSTANCE_ACTIVE

    def is_going_to_active(self):
        return self.target_state == self.INSTANCE_ACTIVE

    def is_prepared(self):
        return self.current_state == self.INSTANCE_PREPARED

    def is_going_to_prepared(self):
        return self.target_state == self.INSTANCE_PREPARED

    def is_removed(self):
        return self.current_state == self.INSTANCE_REMOVED

    def is_going_to_removed(self):
        return self.target_state == self.INSTANCE_REMOVED

    def __cmp__(self, other):
        return cmp((self.configuration_id, self.host, self.service),
                   (other.configuration_id, other.host, other.service))


class IssInstanceHistoryRecord(object):
    """
    Запись про изменение состояния инстанса
    """

    def __init__(self, configuration_id, host, service, timestamp, state,
                 properties=None, text_properties=None, message=None, author=None):
        """
        :param configuration_id: идентификатор конфигурации
        :param host: название хоста
        :param service: название сервиса
        :param timestamp: время изменения состояния в секундах
        :param state: изменение состояния
        :param properties: словарь с информацией про хост при изменении
        :param text_properties: строковое представление поля properties; присутствует только при
                                указанном параметре dump_format (например, со значением "JSON")
        :param message: сообщение при изменении
        :param author: автор изменения
        """
        self.instance = IssInstance(configuration_id, host, service)
        self.timestamp = timestamp
        self.state = state
        self.properties = properties or {}
        self.text_properties = text_properties
        self.message = message
        self.author = author

    def __str__(self):
        return 'Instance of configuration {} on slot {} change state on {} to {}'.format(
            self.instance.configuration_id, self.instance.slot, self.timestamp, self.state)

    def __repr__(self):
        return 'IssInstanceHistoryRecord({}, {}, {}, {}, {}, {}, {}, {}, {})'.format(
            self.instance.configuration_id, self.instance.host, self.instance.service, self.timestamp, self.state,
            self.properties, self.text_properties, self.message, self.author
        )


class IIssApiWrapper(object):
    """
    Интерфейс для inject
    """

    @classmethod
    def instance(cls):
        """
        :rtype: IIssApiWrapper
        """
        return inject.instance(cls)

    def get_instance_properties(self, configuration_id, slot, dump_format='JSON'):
        """
        Возвращаетт свойства инстанса (их содержимое близко к dump.json)

        Формат возвращаемых данных определяется в :param dump_format:, допустимые значения:
        "JSON" и None.

        :type configuration_id: str | unicode
        :type slot: str | unicode
        :type dump_format: str | unicode | NoneType
        :rtype: str
        """
        raise NotImplementedError

    def get_current_state_history(self, slot, since_time, till_time=None, limit=1000, dump_format=None):
        """
        Возвращает историю изменение текущего состояния на указанном слоте за указанный промежуток времени.

        :param slot: идентификатор слота
        :param since_time: начало промежутка, за который нужно получить записи
        :param till_time: завершение промежутка, за который нужно получить записи.
            По умолчанию равен времени на момент вызова.
        :param limit: ограничение по количеству возвращаемых записей
        :param dump_format: формат словаря с properties (см. ISS-2401). Допустимые значения: JSON.
        :rtype: list[IssInstanceHistoryRecord]
        """
        raise NotImplementedError

    @staticmethod
    def get_configuration_id_from_name(configuration_name, family=None):
        """
        Сформировать идентификатор конфигурации из её названия

        :param configuration_name: название конфигурации
        :param family: название семейства конфигураций. Если не задано, вычисляется из configuration_name
        :type configuration_name: str | unicode
        :type family: str | unicode

        :return: идентификатор конфигурации
        :rtype: str | unicode
        """
        raise NotImplementedError

    def list_configuration_instances_raw(self, configuration_id):
        """
        Получить полное описание инстансов конфигурации

        :type configuration_id: unicode | str
        :rtype: list[iss_types.Instance]
        :raise: iss_types.NotFoundException
        """
        raise NotImplementedError

    def list_configuration_slots_raw(self, configuration_id):
        """
        Получить полное описание инстансов конфигурации

        :type configuration_id: unicode | str
        :rtype: list[str | unicode]
        """
        raise NotImplementedError

    def set_target_state_raw(self, instance_filter, target_state, message):
        """
        Sets target state for arbitrary instance filter.

        :type instance_filter: unicode
        :type target_state: unicode
        :type message: unicode
        :rtype: iss_thrift3.ttypes.ApplyResponse
        """
        raise NotImplementedError


class IIssApiWrapperFactory(object):

    @classmethod
    def instance(cls):
        """
        :rtype: IIssApiWrapperFactory
        """
        return inject.instance(cls)

    def get_api_wrapper(self, iss_id):
        """
        :param iss_id: Название инстанса звездолета(iss, iss_admin, ...)
        :type iss_id: unicode
        :rtype: IIssApiWrapper
        """
        raise NotImplementedError

    def list_installation_ids(self):
        """
        :rtype: list[unicode]
        """
        raise NotImplementedError


class IssApiWrapperFactory(IIssApiWrapperFactory):

    def __init__(self, **kwargs):
        self._iss_instances = {}
        for iss_id, config in six.iteritems(kwargs):
            self._iss_instances[iss_id] = IssApiWrapper(**config)

    def get_api_wrapper(self, iss_id):
        return self._iss_instances.get(iss_id, None)

    def list_installation_ids(self):
        return list(six.iterkeys(self._iss_instances))


class IssApiWrapper(IIssApiWrapper, IssStates):
    """
    Обёртка для работы с API кешера ISS
    """

    # адрес API сервиса keyvalue https://planner.yandex-team.ru/projects/16178
    # используется для хранения файлов топологии и филлеров
    KEYVALUE_API_URL = 'http://keyvalue.qe.yandex-team.ru/api/versioned/keyvalue/'

    # путь для предоставления данных в стат ручке
    METRICS_PATH_NAME = ('clients', 'iss')

    # таймаут на один запрос к iss
    REQUEST_TIMEOUT = 120
    # сколько делать повторных попыток
    MAX_TRIES = 5
    # регулировка интенсивности
    MAX_DELAY = 30

    # значения по умолчанию для порций объектов
    DEFAULT_OFFSET = 0
    DEFAULT_LIMIT = 2147483647

    # строчка для описания класса, используется при описании инстанса для ISS
    ISS_INSTANCE_CLASS_NAME = 'ru.yandex.iss.Instance'

    # названия стандартных бранчей
    CURRENT_BRANCH = 'CURRENT'
    PREVIOUS_BRANCH = 'PREVIOUS'

    def __init__(self, host='iss3.yandex-team.ru', port=9090, default_user='loadbase', request_timeout=10, max_tries=3,
                 idle_method=gevent.sleep, idle_byte_period=None, metrics_path_name=None,
                 concurrent_request_count=None, conn_maxsize=None,
                 use_ssl=False, ca_certs=None, certfile=None, connection_timeout_ms=None):
        """
        :param str host: название хоста API звездолёта
        :param int port: порт API звездолёта
        :param str default_user: название пользователя, от кого делать изменения в ISS
        :param int request_timeout: таймаут на API вызов
        :param int max_tries: сколько раз делать перезапрос на API вызов
        :type idle_method: collections.Callable
        :type idle_byte_period: int
        :type metrics_path_name: unicode
        :param int concurrent_request_count: number of concurrent api requests
        :param int conn_maxsize: maxsize of thrift connection pool
        :param bool use_ssl: use ssl or not
        :param str ca_certs: filename to the Certificate Authority pem file
        :param str certfile: filename to certificate pem file
        :type connection_timeout_ms: int
        """
        self.host = host
        self.port = port
        self.log = logging.getLogger(__name__ + '.' + self.__class__.__name__)
        if metrics_path_name:
            metrics_path = [metrics_path_name]
        else:
            metrics_path = self.METRICS_PATH_NAME
        self.iss_metrics = metrics.ROOT_REGISTRY.path(*metrics_path)
        self.default_user = str(default_user)
        self._iss = None
        self.request_timeout = request_timeout
        self.max_tries = max_tries
        self.idle_method = idle_method
        self.idle_byte_period = idle_byte_period
        self.concurrent_request_count = concurrent_request_count
        self.conn_maxsize = conn_maxsize
        self.use_ssl = use_ssl
        self.ca_certs = ca_certs
        self.certfile = certfile
        self.connection_timeout_ms = connection_timeout_ms

    @property
    def iss(self):
        """

        :return: Обёртка для работы с ISS API
        :rtype: IssService.Client
        """
        if self._iss is None:
            self._iss = ThriftBinClient(
                IssService, self.host, self.port, self.request_timeout, self.max_tries, self.iss_metrics,
                idle_method=self.idle_method, idle_byte_period=self.idle_byte_period,
                concurrent_request_count=self.concurrent_request_count, conn_maxsize=self.conn_maxsize,
                use_ssl=self.use_ssl, ca_certs=self.ca_certs, certfile=self.certfile,
                connection_timeout_ms=self.connection_timeout_ms,
            )
        return self._iss

    @staticmethod
    def get_family_from_configuration_id(configuration_id):
        """
        Получить название семейства (family) из идентификатора конфигурации

        :param configuration_id: идентификатор конфигурации
        :return: название семейства
        :rtype: str
        """
        return IssApiWrapper.check_configuration_id(configuration_id).split('#', 2)[0]

    @staticmethod
    def get_name_from_configuration_id(configuration_id):
        """
        Получить название конфигурации из идентификатора конфигурации

        :param configuration_id: идентификатор конфигурации
        :return: название конфигурации
        :rtype: str
        """
        return IssApiWrapper.check_configuration_id(configuration_id).split('#', 2)[1]

    def get_version(self):
        """
        Получить версию ISS кешера (API)

        :return: объект с описанием версии ISS кешера
        :rtype: IssCacherVersion
        """
        result = self.iss.getVersion()
        return IssCacherVersion(result.version.version, result.version.timestamp, result.signature.hostFQDN)

    @staticmethod
    def _get_portion_object(offset=None, limit=None):
        """
        Получить объект Portion

        :param offset: смещение в выборке
        :param limit: максимальное количество возвращаемых объектов
        :return: объект ограничения выборки
        :rtype: iss_types.Portion
        """
        if not offset:
            offset = IssApiWrapper.DEFAULT_OFFSET
        if not limit:
            limit = IssApiWrapper.DEFAULT_LIMIT
        return iss_types.Portion(offset=offset, limit=limit)

    @staticmethod
    def check_slot_name(slot_name):
        """
        Проверить название слота

        :return: название переданного слота, если он прошёл проверку; иначе вызывается IssValidationError
        :rtype: str
        """
        if not len(slot_name.split('@')) == 2:
            raise IssValidationError('Incorrect slot name format for slot {}'.format(slot_name))
        else:
            return slot_name

    @staticmethod
    def check_configuration_id(configuration_id):
        """
        Проверить идентификатор конфигурации

        :param str | unicode configuration_id: идентификатор конфигурации
        :return: идентификатор конфигурации. Если он неправильный, вызывает IssValidationError
        :rtype: str
        """
        # ISS хочет идентификаторы длиной не меньше 10 символов
        if len(str(configuration_id)) > 10 and ('#' in configuration_id):
            return configuration_id
        else:
            raise IssValidationError('Incorrect configuration id format {}'.format(configuration_id))

    @staticmethod
    def get_configuration_name_from_id(configuration_id):
        """
        Получить имя конфигурации из её идентификатора

        :param str configuration_id: идентификатор конфигурации
        :return: имя конфигурации
        :rtype: str
        """
        return IssApiWrapper.check_configuration_id(configuration_id).split('#', 2)[1]

    @staticmethod
    def _get_instance_object(slot, properties=None):
        """
        Получить объект Instance по переданным данным

        :param str slot: слот инстанса (в формате {service}@{hostname}
        :param dict or None properties: свойства инстанса
        :return: объект Instance
        :rtype: iss_types.Instance
        """
        return iss_types.Instance(slot=IssApiWrapper.check_slot_name(slot), properties=properties)

    def _get_signature_object(self, message, author=None):
        """
        Вернуть объект подписи по переданным автору и лог-сообщению

        :param message: сообещние для подписи
        :param author: автор для подписи
        :return: объект подписи
        :rtype: iss_types.AuthorSignature
        """
        if author is None:
            author = self.default_user
        return iss_types.AuthorSignature(author=author, message=message)

    def set_configuration_target_state(self, configuration_id, target_state, slot=None):
        """
        Установить для указанной конфигурации указанное целевое состояние.
        После этого ISS будет пробовать перевести конфигурацию в указанное состояние

        :param configuration_id: идентификатор конфигурации
        :param target_state: состояние, которое будет задано
        :param slot: конкретный слот, где нужно установить состояние конфигурации
        :return: список затронутых инстансов
        :rtype: list of IssInstance
        """
        iss_result = self.set_configuration_target_state_raw(configuration_id, target_state, slot)
        result = []
        for configuration_id, slots in six.iteritems(iss_result.affected):
            for slot in slots:
                result.append(self.get_instance_object_from_configuration_id_and_slot(
                    configuration_id, slot
                ))
        return result

    def set_configuration_target_state_raw(self, configuration_id, target_state, slot, message=None):
        """
        Установить для указанной конфигурации указанное целевое состояние.
        После этого ISS будет пробовать перевести конфигурацию в указанное состояние

        :param configuration_id: идентификатор конфигурации
        :param target_state: состояние, которое будет задано
        :param slot: конкретный слот, где нужно установить состояние конфигурации
        :type message: unicode
        :rtype: iss_types.ApplyResponse
        """
        configuration_id = self.check_configuration_id(configuration_id)
        if message is None:
            message = 'Set configuration {conf} target state to {state}'.format(conf=configuration_id,
                                                                                state=target_state)
            if slot:
                message += 'on slot {slot}'.format(slot=slot)

        if slot:
            instance_filter = "instance.slot=='{slot}'".format(slot=slot)
        else:
            instance_filter = "any-instance"
        iss_filter = "(configuration.id=='{conf}') && ({instance_filter})".format(
            conf=configuration_id, instance_filter=instance_filter)
        return self.set_target_state_raw(instance_filter=iss_filter,
                                         target_state=target_state,
                                         message=message)

    def set_target_state_raw(self, instance_filter, target_state, message):
        signature = self._get_signature_object(message)
        # здесь consistencyLevel -- это consistency level в Cassandra
        # token -- это security token, использование которого в iss пока не имплементировано
        return self.iss.apply(
            transition=iss_types.TargetStateTransition(filters=[instance_filter], state=target_state),
            signature=signature,
            consistencyLevel=iss_types.ConsistencyLevel.DEFAULT,
            token=None
        )

    def get_instance_state(self, configuration_id, slot, get_details=False):
        """
        Получить текущее состояние конфигурации
        Используем getInstanceState по умолчанию и getDetailedInstanceState, если нужны подробности.
        Например, при фейле.

        :param configuration_id: идентификатор конфигурации
        :param slot: слот инстанса
        :param get_details: получить подробности
        :return: объект инстанса или None, если инстанс не найден
        :rtype: IssInstance or None
        """
        result = None
        configuration_id = self.check_configuration_id(configuration_id)
        instance_id = iss_types.InstanceId(slot=slot, configuration=configuration_id)
        try:
            if get_details:
                iss_response = self.iss.getDetailedInstanceState(instance_id, dumpFormat=None)
                target_state = iss_response.instance.target
                current_state = iss_response.instance.feedback.state
                properties = iss_response.instance.feedback.properties
            else:
                iss_response = self.iss.getInstanceState(instance_id)
                target_state = iss_response.instance.targetState
                current_state = iss_response.instance.feedbackState
                properties = {}
            result = self.get_instance_object_from_configuration_id_and_slot(
                iss_response.instance.id.configuration, iss_response.instance.id.slot
            )
            result.target_state = target_state
            result.current_state = current_state
            result.properties = properties
        except iss_types.NotFoundException:
            # если инстанс не найден, ничего не пробрасываем дальше
            pass
        return result

    def get_instance_state_raw(self, configuration_id, slot):
        """
        Получить текущее состояние конфигурации
        Используем getInstanceState.

        :param configuration_id: идентификатор конфигурации
        :param slot: слот инстанса
        :rtype: iss_types.GetInstanceResponse
        """
        configuration_id = self.check_configuration_id(configuration_id)
        instance_id = iss_types.InstanceId(slot=slot, configuration=configuration_id)
        return self.iss.getInstanceState(instance_id)

    def prepare_configuration(self, configuration_id, slot=None):
        """
        Подготовить конфигурацию, то есть задать целевое состояние PREPARED

        :param configuration_id: идентификатор конфигурации
        :param slot: выполнить только для указанного слота
        :return: список затронутых инстансов
        :rtype: list of IssInstance
        """
        configuration_id = self.check_configuration_id(configuration_id)
        return self.set_configuration_target_state(configuration_id, self.INSTANCE_PREPARED, slot)

    def prepare_configuration_raw(self, configuration_id, slot=None):
        """
        Подготовить конфигурацию, то есть задать целевое состояние PREPARED

        :param configuration_id: идентификатор конфигурации
        :param slot: выполнить только для указанного слота
        :rtype: iss_types.ApplyResponse
        """
        configuration_id = self.check_configuration_id(configuration_id)
        return self.set_configuration_target_state_raw(configuration_id, self.INSTANCE_PREPARED, slot)

    def activate_configuration(self, configuration_id, slot=None):
        """
        Активировать конфигурацию, то есть поставить в качестве целевого состояния ACTIVATE

        :param configuration_id: идентификатор конфигурации
        :param slot: выполнить только для указанного хоста
        :return: список затронутых инстансов
        :rtype: list of IssInstance
        """
        configuration_id = self.check_configuration_id(configuration_id)
        return self.set_configuration_target_state(configuration_id, self.INSTANCE_ACTIVE, slot)

    def activate_configuration_raw(self, configuration_id, slot=None):
        """
        Активировать конфигурацию, то есть поставить в качестве целевого состояния ACTIVATE

        :param configuration_id: идентификатор конфигурации
        :param slot: выполнить только для указанного хоста
        :rtype: iss_types.ApplyResponse
        """
        configuration_id = self.check_configuration_id(configuration_id)
        return self.set_configuration_target_state_raw(configuration_id, self.INSTANCE_ACTIVE, slot)

    def remove_configuration(self, configuration_id, slot=None):
        """
        Удалить конфигурацию, то есть поставить в качестве целевого состояния REMOVED

        :param configuration_id: идентификатор конфигурации
        :param slot: конкретный слот, где нужно удалить конфигурацию
        :return: список затронутых инстансов
        :rtype: list of IssInstance
        """
        configuration_id = self.check_configuration_id(configuration_id)
        return self.set_configuration_target_state(configuration_id, self.INSTANCE_REMOVED, slot)

    def create_configuration_from_dump(self, configuration_id, dump, dump_format='JSON', is_applicable=True,
                                       properties=None, message=None, author=None, compress_conf_dump=True):
        """
        Создать конфигурацию в ISS из дампа конфигурации

        :param configuration_id: идентификатор конфигурации
        :type configuration_id: str | unicode
        :param dump: дамп конфигурации
        :type dump: str | unicode
        :param dump_format: формат дампа: "CMS" или "JSON"
        :type dump_format: str | unicode
        :param is_applicable: нужно ли делать конфигурацию применяемой
        :type is_applicable: bool
        :param properties: параметры для всей конфигурации в целом
        :type properties: dict[str | unicode, str | unicode]
        :param message: сообщение, связываемое с конфигурацией
        :type message: str | unicode
        :param author: имя автора конифигурации
        :type author: str | unicode
        :param compress_conf_dump: сжимать ли дамп конфигурации перед отправкой в ISS
        :type compress_conf_dump: bool
        :return: идентификатор созданной конфигурации
        :rtype: unicode
        """
        configuration_id = self.check_configuration_id(configuration_id)
        family = self.get_family_from_configuration_id(configuration_id)
        if not message:
            message = 'Create configuration {}'.format(configuration_id)
        signature = self._get_signature_object(message, author)

        compression_method = 'NONE'

        if compress_conf_dump:
            dump = zlib.compress(dump)
            compression_method = 'DEFLATE'

        content = iss_types.InlineContent(data=dump,
                                          format=dump_format,
                                          compressionMethod=compression_method)

        response = self.iss.createConfigurationFromContent(
            family=family,
            dump=content,
            signature=signature,
            isApplicable=is_applicable,
            configurationId=configuration_id,
            properties=properties
        )

        return response.configuration.id

    def create_configuration(
            self, configuration_id, instances, is_applicable=True, properties=None, message=None, author=None
    ):
        """
        Создать конфигурацию в ISS.

        :param str configuration_id: идентификатор конфигурации
        :param str instances: список описаний инстансов
        :param bool is_applicable: нужно ли делать конфигурацию применяемой
        :param dict properties: параметры для всей конфигурации в целом
        :param str message: объект подписи для создания конфигурации
        :param str author: имя автора конифигурации
        :return: идентификатор созданной конфигурации
        :rtype: str
        """
        configuration_id = self.check_configuration_id(configuration_id)
        family = self.get_family_from_configuration_id(configuration_id)
        if not message:
            message = 'Create configuration with is {} from {} instances'.format(configuration_id, len(instances))
        signature = self._get_signature_object(message, author)
        response = self.iss.createConfiguration(
            family, instances, signature, is_applicable, configuration_id, properties
        )
        return response.configuration.id

    def create_configuration_from_external(
            self, configuration_id, instances_url, is_applicable=True, properties=None, message=None, author=None
    ):
        """
        Создать конфигурацию по JSON дампу инстансов

        :param str configuration_id: идентификатор конфигурации
        :param str instances_url: url с дампом инстансов в формате json
        :param bool is_applicable: нужно ли делать конфигурацию применяемой
        :param dict properties: параметры для всей конфигурации в целом
        :param str message: объект подписи для создания конфигурации
        :param str author: имя автора конифигурации
        :return: идентификатор созданной конфигурации
        :rtype: str
        """
        configuration_id = self.check_configuration_id(configuration_id)
        if not message:
            message = 'Create configuration with is {} from {}'.format(configuration_id, instances_url)
        signature = self._get_signature_object(message, author)
        family = self.get_family_from_configuration_id(configuration_id)
        response = self.iss.createConfigurationFromExternal(
            family, instances_url, 'JSON', signature, is_applicable, configuration_id, properties
        )
        return response.configuration.id

    def create_configuration_from_cms_dump(
            self, name, topology_uri, fillers_uri, family, is_applicable=True,
            properties=None, message=None, author=None
    ):
        """
        Создать конфигурацию в ISS

        :param name: название конфигурации
        :param topology_uri: ссылка на CMS dump файл топологии
        :param fillers_uri: ссылка на json файл филлеров
        :param family: семейство конфигурации
        :param bool is_applicable: нужно ли делать конфигурацию применяемой
        :param dict properties: параметры для всей конфигурации в целом
        :param str message: объект подписи для создания конфигурации
        :param str author: имя автора конифигурации
        :return: идентификатор созданной конфигурации
        :rtype: str
        """
        if message is None:
            message = 'new configuration in family {family}'.format(family=family)
        signature = self._get_signature_object(message, author)
        # ставим isApplicable в False, так как это по сути топология, её поднимать не будем
        # передаём name как идентификатор
        conf = self.iss.createConfigurationFromExternal(
            family, topology_uri, 'CMS', signature, False, None, properties
        )
        parent_configuration_id = conf.configuration.id
        # передаём name как идентификатор
        configuration_id = self.get_configuration_id_from_name(name)
        result_configuration = self.iss.copyAndModifyConfigurationFromExternal(
            parent_configuration_id, fillers_uri, 'JSON', signature, is_applicable, configuration_id, {}
        ).configuration
        return result_configuration.id

    @staticmethod
    def get_family_from_configuration_name(configuration_name):
        """
        Получить название семейства из названия конфигурации

        :param configuration_name: название конфигурации
        :return: название семейства
        :rtype: str
        """
        return configuration_name.split('-')[0]

    def set_configuration_tag(self, configuration_id, branch):
        """
        Назначить бранч конфигурации (подвинуть бранч)

        :param str | unicode configuration_id: идентификатор конфигурации
        :param str | unicode branch: название бранча
        :return: название назначенного бранча
        :rtype: str
        """
        configuration_id = self.check_configuration_id(configuration_id)
        message = 'Mark configuration {conf} as branch {branch}'.format(conf=configuration_id, branch=branch)
        signature = self._get_signature_object(message)
        self.iss.setBranchHead(configuration_id, branch, signature)
        return branch

    def remove_configuration_tag(self, family, branch):
        """
        Убрать бранч конфигурации (подвинуть бранч)

        :param str family: название семейства конфигураций
        :param str branch: название бранча
        :return: название удалённого бранча
        :rtype: str
        """
        message = 'Remove branch {} for family {}'.format(branch, family)
        signature = self._get_signature_object(message)
        self.iss.removeBranchHead(family, branch, signature)
        return branch

    def remove_configuration_tag_by_configuration(self, configuration_id, branch):
        """
        Убрать бранч конфигурации (подвинуть бранч)

        :param str configuration_id: идентификатор конфигурации
        :param str branch: название бранча
        :return: название назначенного бранча
        :rtype: str
        """
        configuration_id = self.check_configuration_id(configuration_id)
        family = self.get_family_from_configuration_id(configuration_id)
        return self.remove_configuration_tag(family, branch)

    def add_current_tag(self, configuration_id):
        """
        Отметить конфигурацию как текущую для семейства

        :param str configuration_id: идентификатор конфигурации
        :return: название назначенного бранча
        :rtype: str
        """
        return self.set_configuration_tag(configuration_id, self.CURRENT_BRANCH)

    def add_previous_tag(self, configuration_id):
        """
        Отметить конфигурацию как предшесвующую текущей для семейства

        :param str configuration_id: идентификатор конфигурации
        :return: название назначенного бранча
        :rtype: str
        """
        return self.set_configuration_tag(configuration_id, self.PREVIOUS_BRANCH)

    def get_current_configuration(self, family):
        """
        Получить идентификатор текущей конфигурации для указанного семейства

        :param str family: название семейства в виде строки
        :return: идентификатор конфигурации в виде строки; Nonе, если конфигурация не была найдена
        :rtype: str or None
        """
        return self.get_configuration_by_family_and_branch(family, self.CURRENT_BRANCH)

    def get_previous_configuration(self, family):
        """
        Получить идентификатор предшествующей текущей конфигурации для указанного семейства

        :param str family: название семейства в виде строки
        :return: идентификатор конфигурации в виде строки; Nonе, если конфигурация не была найдена
        :rtype: str or None
        """
        return self.get_configuration_by_family_and_branch(family, self.PREVIOUS_BRANCH)

    def get_configuration_by_family_and_branch(self, family, branch):
        """
        Получить идентификатор конфигурации для указанного семейства и ветки

        :param str | unicode family: название семейства в виде строки
        :param str | unicode branch: название бранча
        :return: идентификатор конфигурации в виде строки; Nonе, если конфигурация не была найдена
        :rtype: str or None
        """
        try:
            return self.iss.getBranchHead(family, branch).configuration
        except iss_types.NotFoundException:
            pass
        return None

    @staticmethod
    def _get_iss_configuration_object(configuration):
        """
        Перевести описание конфигурации в dict

        :param configuration: thrift-объект конфигурации
        :return: объект с описанием конфигурации
        :rtype: IssConfiguration
        """
        return IssConfiguration(
            configuration_id=configuration.id,
            parent=configuration.parent,
            created=configuration.timestamp,
            is_applicable=configuration.isApplicable,
            properties=configuration.properties,
        )

    def get_configuration_by_id(self, configuration_id):
        """
        Получить описание конфигурации по её идентификатору

        :param str | unicode configuration_id: идентификатор конфигурации
        :return: объект с описанием конфигурации
        :rtype: IssConfiguration
        """
        configuration_id = self.check_configuration_id(configuration_id)
        try:
            conf = self.iss.getConfigurationDescription(configuration_id).configuration
            return self._get_iss_configuration_object(conf)
        except iss_types.NotFoundException:
            return None

    @staticmethod
    def get_configuration_id_from_name(configuration_name, family=None):
        if not configuration_name:
            # FIXME: WTF?
            return None
        if family is None:
            family = IssApiWrapper.get_family_from_configuration_name(configuration_name)
        return '{family}#{name}'.format(family=family, name=configuration_name)

    def get_configurations_by_branch_name(self, branch_name):
        """
        Получить список идентификаторов конфигураций по названию бранча

        :param str | unicode branch_name: название бранча
        :return: список идентификаторов конфигураций
        :rtype: list of str
        """
        return [str(configuration) for configuration in self.iss.getBranchHeads(branch_name).configurations]

    def get_configuration_by_name(self, configuration_name):
        """
        Получить идентификатор конфигурации по её имени

        :param str | unicode configuration_name: название (имя) конфигурации
        :return: идентификатор конфигурации
        :rtype: str
        """
        # пробуем получить конфигурацию по name как id
        configuration_id = self.get_configuration_by_id(self.get_configuration_id_from_name(configuration_name))
        if configuration_id:
            return configuration_id.id

        # для обратной совместимости всё ещё пробуем получить конфигурацию через ветку
        try:
            branch_configurations = self.get_configurations_by_branch_name(configuration_name)
            if not branch_configurations:
                return None
            else:
                # выбираем первую кофнигурацию (по идее, она там и должна быть одна)
                configuration_id = branch_configurations[0]
                return configuration_id
        except iss_types.NotFoundException:
            return None

    @staticmethod
    def get_host_and_service_from_slot(slot):
        """
        Получить порт и имя хоста из имени сервиса (в терминах bsconfig - порта)

        :return: пару host, service
        :rtype: tuple
        """
        service, host = slot.split('@', 2)
        return host, service

    @staticmethod
    def get_slot_name_from_host_and_service(host, service):
        """
        Получить имя сервиса (в терминах bsconfig - порта) и имя хоста из названия слота

        :return: название слота
        :rtype: str | unicode
        """
        return '{}@{}'.format(service, host)

    def get_slot_instances(self, slot):
        """
        Получить список инстансов для указанного слота

        :param str | unicode slot: идентификатор слота в формате {slot_name}@{hostname}
        :return: идентификатор конфигурации, если что-то поднято, None в противном случае
        :rtype: list of IssInstance
        """
        result = []
        try:
            response = self.iss.getSingleSlotConfiguration(self.check_slot_name(slot))
            if response:
                for iss_instance in response.instances:
                    instance = self.get_instance_object_from_configuration_id_and_slot(
                        iss_instance.id.configuration, iss_instance.id.slot
                    )
                    instance.target_state = iss_instance.targetState
                    instance.properties = iss_instance.properties
                    result.append(instance)
        except iss_types.NotFoundException:
            pass
        return result

    def get_slot_active_configuration(self, slot):
        """
        Получить идентификатор активной конфигурации на слоте

        :param str | unicode slot: идентификатор слота в формате {slot_name}@{hostname}
        :return: идентификатор конфигурации, если что-то поднято, None в противном случае
        :rtype: string or None
        """
        try:
            # ищем ACTIVE конфигурацию и возвращаем её id
            for instance in self.get_slot_instances(slot):
                if instance.is_going_to_active():
                    return instance.configuration_id
        except iss_types.NotFoundException:
            pass
        return None

    def list_family_configurations(self, family):
        """
        Получить список идентификаторов конфигураций по названию семейства

        :param str | unicode family: название семейства
        :return: список строк-идентификаторов конфигураций. Если не найдено, возвращается пустой список
        :return: list of str
        """
        return [str(configuration) for configuration in self.iss.getFamilyConfigurations(family).configurations]

    def list_families(self, family_filter=None):
        """
        Получить список доступных семейств

        :param str | unicode | None family_filter: фильтр в виде regexp выражения
        :return: список названий семейств в виде строк
        :rtype: list of str
        """
        return [str(family) for family in self.iss.getConfigurationFamilies(family_filter).families]

    def destroy_configuration(self, configuration_id):
        """
        Уничтожить конфигурацию (убрать из кешера навсегда)

        :param str | unicode configuration_id: идентификатор конфигурации
        :rtype: bool
        """
        configuration_id = self.check_configuration_id(configuration_id)
        message = 'Destroy configuration {configuration_id}.'.format(
            configuration_id=configuration_id
        )
        result = self.iss.destroyConfiguration(configuration_id, self._get_signature_object(message), None)
        return bool(result.destroyed)

    def get_instance_object_from_configuration_id_and_slot(self, configuration_id, iss_instance):
        """
        Получить объект инстанса по идентификатору конфигурации и строковому представлению слота

        :param str | unicode configuration_id: идентификатор конфигурации
        :param str | unicode iss_instance: строковое представление слота
        :return:
        """
        configuration_id = self.check_configuration_id(configuration_id)
        host, service = self.get_host_and_service_from_slot(iss_instance)
        return IssInstance(configuration_id, host, service)

    def list_configuration_instances(self, configuration_id, offset=None, limit=None):
        """
        Получить список слотов инстансов для конфигурации

        :param str | unicode configuration_id: идентификатор конфигурации
        :param int | None offset: смещение для выборки
        :param int | None limit: максимальное количество объектов в выборке
        :rtype: list of IssInstance
        """
        configuration_id = self.check_configuration_id(configuration_id)
        result = []
        iss_response = self.iss.query(
            "(configuration.id == '{conf_id}') && (any-instance)".format(conf_id=configuration_id),
            self._get_portion_object(offset, limit)
        )
        if iss_response.result:
            for iss_result in iss_response.result:
                for instance in iss_result.instances:
                    result.append(self.get_instance_object_from_configuration_id_and_slot(
                        iss_result.configuration, instance
                    ))
        return result

    def list_configuration_instances_raw(self, configuration_id):
        return self.iss.getConfiguration(configuration_id, self._get_portion_object(), dumpFormat=None).instances

    def list_configuration_slots_raw(self, configuration_id):
        query = "(configuration.id == '{}') && (any-instance)".format(configuration_id)
        result = self.iss.query(query, portion=self._get_portion_object()).result
        return result[0].instances if result else []

    def get_configuration_instances_with_resources(self, configuration_id, offset=None, limit=None):
        """
        Получить список объектов инстансов по идентификатору конфигурации с информацией о ресурсах, шардах и т.д.

        :param str | unicode configuration_id: идентификатор конфигурации
        :param int | None offset: смещение для выборки
        :param int | None limit: максимальное количество объектов в выборке
        :rtype: list of IssInstance
        """
        configuration_id = self.check_configuration_id(configuration_id)
        result = []
        iss_response = None
        try:
            iss_response = self.iss.getConfiguration(configuration_id, self._get_portion_object(offset, limit),
                                                     dumpFormat=None)
        except iss_types.NotFoundException:
            # если конфигурация не найдена, ничего не пробрасываем дальше
            pass
        if iss_response and iss_response.instances:
            for instance_info in iss_response.instances:
                instance = self.get_instance_object_from_configuration_id_and_slot(configuration_id, instance_info.slot)
                instance.properties = instance_info.properties
                result.append(instance)
        return result

    def get_host_instances(self, hostname):
        """
        Получить список инстансов хоста

        :param hostname: полное имя хоста
        :return: список инстансов, которые связаны с данным хостом
        :rtype: list of IssInstance
        """

        return self.get_host_configuration(hostname).instances

    def get_host_configuration(self, hostname, footprint=None, removed=False):
        """
        Получаем конфигурацию хоста

        :param str hostname: fqdn
        :param str|None footprint: имеющийся футпринт конфигурации хоста
        :param bool removed: Возвращать инстансы в состоянии REMOVED
        :rtype: IssHostConfiguration
        :raises: iss_thrift3.ttypes.HostConfigurationUnchangedException
        """
        instances = []
        if removed:
            response = self.iss.getHostConfigurationWithRemoved(hostname, footprint)
        else:
            response = self.iss.getHostConfiguration(hostname, footprint)

        for iss_instance in response.instances:
            instance = self.get_instance_object_from_configuration_id_and_slot(
                iss_instance.id.configuration, iss_instance.id.slot
            )
            instance.target_state = iss_instance.targetState
            instance.properties = iss_instance.properties
            instances.append(instance)
        return IssHostConfiguration(instances, response.hostConfigurationFootprint)

    @staticmethod
    def validate_keyvalue_data_name(data_name):
        """
        Проверить корректность url-а для keyvalue

        :param data_name: название данных в keyvaue
        :return: url, если он корректнен; в противном случае вызываем исключение
        :rtype: str
        """
        if (not data_name) or ('/' in data_name):
            raise IssValidationError('Incorrect data name for keyvalue: {data_name}'.format(data_name=data_name))
        return data_name

    @staticmethod
    def upload_from_file_to_keyvalue(file_path, data_name):
        """
        Загрузить из файла в keyvalue-хранилище

        :param file_path: путь до файла с данными
        :param data_name: название данных в keyvalue. Не должно содержать символа '\'
        :return: url по которому доступны данные
                 Если передано некооректное значение для названия файла в keyvalue, вызвается ислкючение.
        :rtype: str
        """
        with open(file_path) as topology_file:
            data = topology_file.read()
        return IssApiWrapper.upload_data_to_keyvalue(data_name, data)

    @staticmethod
    def upload_data_to_keyvalue(data_name, data_string):
        """
        Загрузить данные в keyvalue по данному названию

        :param data_name: название данных
        :param data_string: данные в виде строки
        :return: url по которому доступны данные
            Если передано некооректное значение для названия файла в keyvalue, вызвается ислкючение.
        :rtype: str
        """
        url = IssApiWrapper.KEYVALUE_API_URL + IssApiWrapper.validate_keyvalue_data_name(data_name)
        requests.post(url, data=data_string)
        return url

    def send_feedback(self, slot, configuration_id, state, properties=None):
        """
        Отослать обратную связь кешеру
        Всегда возвращает True, если ISS что-то ответил:
        https://st.yandex-team.ru/ISS-1886

        :param slot: по какому слоту
        :param configuration_id: идентификатор конфигурации
        :param state: описание состояния (например, ACTIVE)
        :param properties: словарь пар строка-строка. Просто сохраняется в стейте
        :return: True, если получилось изменить; False в противном случае
        :rtype: bool
        """
        configuration_id = self.check_configuration_id(configuration_id)
        properties = properties or {}
        iss_instance = iss_types.InstanceId(slot=self.check_slot_name(slot), configuration=configuration_id)
        report = iss_types.DetailedCurrentState(
            timestamp=int(time.time()),
            state=str(state),
            properties=properties,
        )
        self.iss.saveFeedback(iss_instance, report)
        return True

    @staticmethod
    def _get_time_frame_object(since_time, till_time=None):
        """
        Получить объект для задания промежутка времени в ISS API

        :param int since_time: начало промежутка в секундах
        :param int till_time: завершение промежутка в секундах
        :return: объект TimeFrame
        :rtype: iss_types.TimeFrame
        """
        if not till_time:
            till_time = int(time.time())
        if till_time < since_time:
            raise IssValidationError('Incorrect time frame values - since {} is bigger than till {}'.format(
                since_time, till_time
            ))
        return iss_types.TimeFrame(since_time * 1000, till_time * 1000)

    def get_current_state_history(self, slot, since_time, till_time=None, limit=1000, dump_format=None):
        slot = self.check_slot_name(slot)
        time_frame = self._get_time_frame_object(since_time, till_time)
        iss_result = self.iss.getCurrentStateHistory(slot, time_frame, limit, dump_format)
        result = []
        for item in iss_result.items:
            host, service_name = self.get_host_and_service_from_slot(item.instanceId.slot)
            result.append(
                IssInstanceHistoryRecord(
                    configuration_id=item.instanceId.configuration,
                    host=host,
                    service=service_name,
                    timestamp=item.timestamp // 1000,
                    state=item.state,
                    properties=item.properties,
                    text_properties=item.textProperties))
        return result

    def get_instance_properties(self, configuration_id, slot, dump_format='JSON'):
        instance = iss_types.InstanceId(slot=slot, configuration=configuration_id)
        response = self.iss.getInstanceProperties(instance, dump_format)
        return response.instance.textProperties

    def get_target_state_history(self, slot, since_time, till_time=None, limit=1000):
        """
        Получить историю изменение целевого состояния на указанном слоте за указанный промежуток времени.

        :param slot: идентификатор слота
        :param since_time: начало промежутка, за который нужно получить записи
        :param till_time: завершение промежутка, за который нужно получить записи.
            По умолчанию равен времени на момент вызова.
        :param limit: ограничение по количеству возвращаемых записей
        :return: список состояний инстансов
        :rtype: list of IssInstanceHistoryRecord
        """
        slot = self.check_slot_name(slot)
        time_frame = self._get_time_frame_object(since_time, till_time)
        iss_result = self.iss.getTargetStateHistory(slot, time_frame, limit)
        result = []
        host, service_name = self.get_host_and_service_from_slot(slot)
        for item in iss_result.items:
            timestamp = item.timestamp // 1000
            for configuration_id, state in six.iteritems(item.transition.configurationTargetState):
                result.append(IssInstanceHistoryRecord(
                    configuration_id=configuration_id,
                    host=host,
                    service=service_name,
                    timestamp=timestamp,
                    state=state,
                    message=item.author.message,
                    author=item.author.author
                ))
        return result

    def toggle_feature(self, feature, enabled, message=None, author=None):
        """
        :param str feature: IssFeatures.*
        :param bool enabled: вкл/выкл
        :param str message: сообщение при изменении
        :param str author: автор изменения
        :return:
        """
        # здесь consistencyLevel -- это consistency level в Cassandra
        # token -- это security token, использование которого в iss пока не имплементировано
        return self.iss.toggleFeature(
            feature=feature,
            enabled=enabled,
            author=self._get_signature_object(message, author),
            consistencyLevel=iss_types.ConsistencyLevel.DEFAULT,
            token=None,
        )
