from functools import partial
import os
import socket
import logging

from kazoo.handlers.gevent import SequentialGeventHandler
from kazoo.client import KazooClient, KazooState
from kazoo.retry import RetryFailedError, KazooRetry
from kazoo.exceptions import (
    ConnectionLossException, OperationTimeoutException, ConnectionClosedError,
    KazooException, NoNodeException, SessionExpiredError, NodeExistsError)

from .recipes import DataWatch, ChildrenWatch, LockingQueue


__all__ = ['CoordError', 'ZookeeperClient',
           'NoNodeException', 'NodeExistsError',
           'LossExceptions', 'KazooState', 'RetryFailedError']


CoordError = KazooException
LossExceptions = (ConnectionLossException,
                  OperationTimeoutException,
                  ConnectionClosedError,
                  SessionExpiredError)


DEFAULT_CONNECTION_RETRY = {
    'max_tries': None,
    'delay': 0.1,
    'backoff': 2,
    'max_jitter': 0.8,
    'max_delay': 60,
}


log = logging.getLogger('zookeeper')


class ZookeeperClient(object):
    """
    Entry point to work with zookeeper:
    * configures connection
    * shares it between users
    * performs all actions in specified chroot
    * provides some highlevel level API
    """

    def __init__(self, cfg, identifier=None, handler=None, metrics_registry=None):
        self.prefix = cfg['zk_root']
        self.hosts = cfg['hosts']
        # kazoo connection spams too much, let's disable it
        if not cfg.get('log_debug', False):
            logging.getLogger('kazoo').setLevel(logging.INFO)
        # zk uses strange protocol to set chroot
        handler = handler or SequentialGeventHandler()
        kwargs = {}
        if getattr(KazooClient, 'IS_SWAT_FLAVOURED', False):
            kwargs['metrics_registry'] = metrics_registry
        self.client = KazooClient(self.hosts + self.prefix, handler=handler,
                                  read_only=cfg.get('read_only', True),
                                  connection_retry=KazooRetry(sleep_func=handler.sleep_func,
                                                              **cfg.get('retry', DEFAULT_CONNECTION_RETRY)),
                                  **kwargs)
        self.identifier = identifier or socket.gethostname()
        self.add_listener(self._session_listener)
        self.LockingQueue = partial(LockingQueue, self.client)

    def _session_listener(self, state):
        log.info("zk connection changed state to {0}".format(state))

    def _stop_client(self):
        try:
            # we may fail to properly stop coordinator
            # kazoo tries to shutdown connection gracefully
            # and throws Exception if fails to do so
            self.client.stop()
            # kazoo puts Close request into writer queue during stop()
            # then, when writer starts again, first thing in queue is Close
            # _reset() call clears queues
            self.client._reset()
        except Exception as e:
            log.error("failed to gracefully stop kazoo: {}".format(str(e)))
            if self.client._connection._socket is not None:
                try:
                    self.client._connection._socket.close()
                except socket.error as e:
                    log.error("failed to close kazoo socket: {}".format(e.strerror))

    # === API to manage lifecycle
    def start(self):
        event = self.client.start_async()
        return event

    def stop(self):
        self._stop_client()

    def close(self):
        self.client.close()

    # === public API
    def add_listener(self, func):
        """
        Add :param func: as session listener.
        Will be called with one argument (Kazoo.State),
        describing new session state when it changes.
        """
        func(self.client.state)
        self.client.add_listener(func)

    def remove_listener(self, func):
        """
        Remove :param func: from session listeners.
        """
        self.client.remove_listener(func)

    @property
    def session_state(self):
        """
        Current zookeeper session state.
        """
        return self.client.state

    def lock(self, path):
        """
        :rtype: Lock
        """
        return self.client.Lock(path, self.identifier)

    def lock_contenders(self, name):
        """
        Handy shortcut to get contenders
        """
        return self.lock(name).contenders()

    def watcher(self, filename):
        """
        Return new watcher for specified node

        Watcher semantics needs clarification.
        We create watcher and pass him callable,
        it immediately tries to read node value and call back.
        If this very first attempt fails - we'll get exception.
        But watcher will be automagically reestablished
        when session reconnects.

        Callback must return True to reestablish watcher
        :param filename: node to watch
        """
        return self.client.DataWatch(filename)

    def data_watcher(self, filename):
        """
        Return new watcher for specified node.
        Alternative implementation of stock watcher with
        same purpose but different API.
        """
        return DataWatch(self.client, filename)

    def ensure_path(self, path):
        return self.client.ensure_path(path)

    def get_children(self, path):
        return self.client.get_children(path)

    def children_watcher(self, catalog):
        """
        callback must return True to reestablish watcher
        """
        return ChildrenWatch(self.client, catalog)

    def write_file(self, filename, value):
        # ensure all but one items in path
        # otherwise will receive 2 change events
        dirname = os.path.dirname(filename)
        self.client.ensure_path(dirname)
        return self.client.set(filename, value)

    def create_file(self, filename, value, acl=None, ephemeral=False, sequence=False):
        return self.client.create(filename, value, makepath=True,
                                  acl=acl, ephemeral=ephemeral, sequence=sequence)

    def read_file(self, filename):
        return self.client.get(filename)

    def delete_file(self, filename, recursive=False):
        return self.client.delete(filename, recursive=recursive)
