"""
Our own recipes implementation for kazoo with
alternative (less error prone) API.
"""
import uuid

from kazoo.client import KazooState
from kazoo.exceptions import NoNodeError, NodeExistsError
from kazoo.protocol.states import EventType
from kazoo.recipe.queue import BaseQueue


class DataWatch(object):
    """Watches a node for data updates"""
    def __init__(self, client, path):
        self._client = client
        self._path = path
        self._watch_established = False
        self._stopped = True
        self._prior = None
        self._wake_event = client.handler.event_object()
        self._client.add_listener(self._session_watcher)

    @property
    def path(self):
        return self._path

    def _get_data(self):
        while 1:
            self._wake_event.wait()
            self._wake_event.clear()
            if self._stopped:
                self._watch_established = False
                break
            data, stat = self._client.retry(self._client.get,
                                            self._path, self._watcher)
            self._watch_established = True
            if self._prior is not None and self._prior.mzxid == stat.mzxid:
                continue
            else:
                self._prior = stat
                return data, stat

    def _watcher(self, _):
        self._watch_established = False
        self._wake_event.set()

    def _session_watcher(self, state):
        if state == KazooState.LOST:
            self._watch_established = False
        elif (state == KazooState.CONNECTED and
              not self._watch_established and
              not self._stopped):
            self._wake_event.set()

    def cancel(self):
        """
        Abort wait.
        """
        if not self._stopped:
            self._stopped = True
            self._wake_event.set()

    def wait(self):
        """
        Wait for data change event.
        :return: tuple with data as string and stats object.
        :rtype: tuple(str, kazoo.client.ZnodeStat)
        :raise: NoNodeException if node does not exists.
        """
        self._stopped = False
        self._wake_event.set()
        return self._get_data()


class ChildrenWatch(object):
    """Watches a node for child list updates"""
    def __init__(self, client, path):
        self._client = client
        self._path = path
        self._watch_established = False
        self._stopped = True
        self._prior = None
        self._wake_event = client.handler.event_object()
        self._client.add_listener(self._session_watcher)

    def _get_children(self):
        while 1:
            self._wake_event.wait()
            self._wake_event.clear()
            if self._stopped:
                self._watch_established = False
                return None
            children = self._client.retry(self._client.get_children,
                                          self._path, self._watcher)
            self._watch_established = True
            if self._prior is not None and self._prior == children:
                continue
            else:
                self._prior = children
                self._stopped = True
                return children

    def _watcher(self, _):
        self._watch_established = False
        self._wake_event.set()

    def _session_watcher(self, state):
        if state == KazooState.LOST:
            self._watch_established = False
        elif (state == KazooState.CONNECTED and
              not self._watch_established and
              not self._stopped):
            self._wake_event.set()

    def cancel(self):
        """
        Abort wait
        """
        if not self._stopped:
            self._stopped = True
            self._wake_event.set()

    def wait(self):
        """
        Wait for data change event.
        :return: list of children node names
        :raise: NoNodeException if node does not exists.
        """
        self._stopped = False
        self._wake_event.set()
        return self._get_children()


class LockingQueue(BaseQueue):
    """A distributed queue with priority and locking support.

    Upon retrieving an entry from the queue, the entry gets locked with an
    ephemeral node (instead of deleted). If an error occurs, this lock gets
    released so that others could retake the entry. This adds a little penalty
    as compared to :class:`Queue` implementation.

    The user should call the :meth:`LockingQueue.get` method first to lock and
    retrieve the next entry. When finished processing the entry, a user should
    call the :meth:`LockingQueue.consume` method that will remove the entry
    from the queue.

    This queue will not track connection status with ZooKeeper. If a node locks
    an element, then loses connection with ZooKeeper and later reconnects, the
    lock will probably be removed by Zookeeper in the meantime, but a node
    would still think that it holds a lock. The user should check the
    connection status with Zookeeper or call :meth:`LockingQueue.holds_lock`
    method that will check if a node still holds the lock.

    """
    lock = "/taken"
    entries = "/entries"
    entry = "entry"

    @staticmethod
    def _filter_locked(values, taken):
        taken = set(taken)
        available = sorted(values)
        return (available if len(taken) == 0 else
                [x for x in available if x not in taken])

    def __init__(self, client, path):
        """
        :param client: A :class:`~kazoo.client.KazooClient` instance.
        :param path: The queue path to use in ZooKeeper.
        """
        super(LockingQueue, self).__init__(client, path)
        self.id = uuid.uuid4().hex.encode()
        self.processing_element = None
        self._lock_path = self.path + self.lock
        self._entries_path = self.path + self.entries
        self.structure_paths = (self._lock_path, self._entries_path)
        self._flag = self.client.handler.event_object()
        self._canceled = False

    def __len__(self):
        """Returns the current length of the queue.

        :returns: queue size (includes locked entries count).
        """
        return super(LockingQueue, self).__len__()

    def put(self, value, priority=100):
        """Put an entry into the queue.

        :param value: Byte string to put into the queue.
        :param priority:
            An optional priority as an integer with at most 3 digits.
            Lower values signify higher priority.

        """
        self._check_put_arguments(value, priority)
        self._ensure_paths()

        self.client.create(
            "{path}/{prefix}-{priority:03d}-".format(
                path=self._entries_path,
                prefix=self.entry,
                priority=priority),
            value, sequence=True)

    def put_all(self, values, priority=100):
        """Put several entries into the queue. The action only succeeds
        if all entries where put into the queue.

        :param values: A list of values to put into the queue.
        :param priority:
            An optional priority as an integer with at most 3 digits.
            Lower values signify higher priority.

        """
        if not isinstance(values, list):
            raise TypeError("values must be a list of byte strings")
        if not isinstance(priority, int):
            raise TypeError("priority must be an int")
        elif priority < 0 or priority > 999:
            raise ValueError("priority must be between 0 and 999")
        self._ensure_paths()

        with self.client.transaction() as transaction:
            for value in values:
                if not isinstance(value, bytes):
                    raise TypeError("value must be a byte string")
                transaction.create(
                    "{path}/{prefix}-{priority:03d}-".format(
                        path=self._entries_path,
                        prefix=self.entry,
                        priority=priority),
                    value, sequence=True)

    def get(self, timeout=None):
        """Locks and gets an entry from the queue. If a previously got entry
        was not consumed, this method will return that entry.

        :param timeout:
            Maximum waiting time in seconds. If None then it will wait
            untill an entry appears in the queue.
        :returns: A locked entry value or None if the timeout was reached.
        :rtype: bytes
        """
        self._canceled = False
        self._flag.clear()
        self._ensure_paths()
        if not self.processing_element is None:  # noqa: E714
            return self.processing_element[1]
        else:
            return self._inner_get(timeout)

    def holds_lock(self):
        """Checks if a node still holds the lock.

        :returns: True if a node still holds the lock, False otherwise.
        :rtype: bool
        """
        if self.processing_element is None:
            return False
        lock_id, _ = self.processing_element
        lock_path = "{path}/{id}".format(path=self._lock_path, id=lock_id)
        self.client.sync(lock_path)
        value, stat = self.client.retry(self.client.get, lock_path)
        return value == self.id

    def consume(self):
        """Removes a currently processing entry from the queue.

        :returns: True if element was removed successfully, False otherwise.
        :rtype: bool
        """
        if not self.processing_element is None and self.holds_lock:  # noqa: E714
            id_, value = self.processing_element
            with self.client.transaction() as transaction:
                transaction.delete("{path}/{id}".format(
                    path=self._entries_path,
                    id=id_))
                transaction.delete("{path}/{id}".format(
                    path=self._lock_path,
                    id=id_))
            self.processing_element = None
            return True
        else:
            return False

    def cancel(self):
        self._canceled = True
        self._flag.set()

    def _inner_get(self, timeout):
        lock = self.client.handler.lock_object()
        canceled = False
        value = []

        def check_for_updates(event):
            if not event is None and event.type != EventType.CHILD:  # noqa: E714
                return
            with lock:
                if canceled or self._flag.isSet():
                    return
                values = self.client.retry(self.client.get_children,
                                           self._entries_path,
                                           check_for_updates)
                taken = self.client.retry(self.client.get_children,
                                          self._lock_path,
                                          check_for_updates)
                available = self._filter_locked(values, taken)
                if len(available) > 0:
                    ret = self._take(available[0])
                    if not ret is None:  # noqa: E714
                        # By this time, no one took the task
                        value.append(ret)
                        self._flag.set()

        check_for_updates(None)
        retval = None
        self._flag.wait(timeout)
        with lock:
            if self._canceled:
                canceled = True
                return None
            canceled = True
            if len(value) > 0:
                # We successfully locked an entry
                self.processing_element = value[0]
                retval = value[0][1]
        return retval

    def _take(self, id_):
        try:
            self.client.create(
                "{path}/{id}".format(
                    path=self._lock_path,
                    id=id_),
                self.id,
                ephemeral=True)
            value, stat = self.client.retry(self.client.get,
                                            "{path}/{id}".format(path=self._entries_path, id=id_))
        except (NoNodeError, NodeExistsError):
            # Item is already consumed or locked
            return None
        return id_, value
