# coding: utf-8
"""TreeCache

:Maintainer: Jiangge Zhang <tonyseek@gmail.com>
:Maintainer: Haochuan Guo <guohaochuan@gmail.com>
:Maintainer: Tianwen Zhang <mail2tevin@gmail.com>
:Status: Alpha

A port of the Apache Curator's TreeCache recipe. It builds an in-memory cache
of a subtree in ZooKeeper and keeps it up-to-date.

See also: http://curator.apache.org/curator-recipes/tree-cache.html
"""

from __future__ import absolute_import

import functools
import logging

import gevent
import gevent.lock
import operator
import os
import six
from kazoo.exceptions import NoNodeError, KazooException
from kazoo.protocol.paths import _prefix_root
from kazoo.protocol.states import KazooState, EventType


logger = logging.getLogger(__name__)


class Codec(object):
    def decode(self, buf, stat=None):
        """
        Decodes bytes into object.

        :type buf: basestring
        :type stat: obj
        """
        raise NotImplementedError

    def encode(self, obj):
        """
        Encodes provided object into bytes.

        :rtype: str
        """
        raise NotImplementedError

    def set_generation(self, obj, generation):
        """
        Sets object's generation.

        :type obj: obj
        :type generation: unicode
        """
        raise NotImplementedError


class Structure(object):
    def get_codec(self):
        """
        :rtype: Codec | None
        """
        raise NotImplementedError

    def get_child(self, path):
        """
        :type path: str
        :rtype: Structure | None
        """
        raise NotImplementedError

    def is_leaf(self):
        raise NotImplementedError


class TreeCache(object):
    """The cache of a ZooKeeper subtree.

    :param client: A :class:`~kazoo.client.KazooClient` instance.
    :param path: The root path of subtree.
    :param Structure structure:
    """

    STATE_LATENT = 0
    STATE_STARTED = 1
    STATE_CLOSED = 2

    def __init__(self, client, path, structure=None):
        self._client = client
        self._root = TreeNode.make_root(self, path, structure=structure)
        self._state = self.STATE_LATENT
        self._is_initialized = False
        self._event_listeners = []

        # https://st.yandex-team.ru/AWACS-982
        # The idea is to put all the *initial* requests to Zookeeper in a separate queue with a separate handler.
        # All subsequent requests (produced by watchers' handlers) have to be put in
        # a regular task queue `self._task_queue`.
        # This way we can reliably detect whether we've performed all initial requests by checking emptiness of our
        # separate `self._initial_task_queue`.
        self._lock = gevent.lock.Semaphore(value=1)
        self._initial_task_queue = client.handler.queue_impl()
        self._initial_task_thread = None
        self._initial_do_background = self._construct_queue_handler_method(self._initial_task_queue, self._lock)
        self._initial_outstanding_ops = 0

        self._task_queue = client.handler.queue_impl()
        self._task_thread = None
        self._do_background = self._construct_queue_handler_method(self._task_queue, self._lock)
        self._outstanding_ops = 0

        self._structure = structure

    @property
    def state(self):
        return self._state

    def start_task_thread(self):
        self._task_thread = self._client.handler.spawn(self._do_background)

    def start_initial_task_thread(self):
        self._initial_task_thread = self._client.handler.spawn(self._initial_do_background)

    def start(self):
        """Starts the cache.

        The cache is not started automatically. You must call this method.

        After a cache started, all changes of subtree will be synchronized
        from the ZooKeeper server. Events will be fired for those activity.

        Don't forget to call :meth:`close` if a tree was started and you don't
        need it anymore, or you will leak the memory of cached nodes, even if
        you have released all references to the :class:`TreeCache` instance.
        Because there are so many callbacks that have been registered to the
        Kazoo client.

        See also :meth:`~TreeCache.listen`.

        .. note::

            This method is not thread safe.
        """
        if self._state == self.STATE_LATENT:
            self._state = self.STATE_STARTED
        elif self._state == self.STATE_CLOSED:
            raise KazooException('already closed')
        else:
            raise KazooException('already started')

        self.start_initial_task_thread()
        self.start_task_thread()
        self._client.add_listener(self._session_watcher)
        self._client.ensure_path(self._root._path)

        if self._client.connected:
            # The on_created and other on_* methods must not be invoked outside
            # the background task. This is the key to keep concurrency safe
            # without lock.
            self._initial_in_background(self._root.on_created)

    def close(self):
        """Closes the cache.

        A closed cache was detached from ZooKeeper's changes. And all nodes
        will be invalidated.

        Once a tree cache was closed, it could not be started again. You should
        only close a tree cache while you want to recycle it.

        .. note::

            This method is not thread safe.
        """
        if self._state == self.STATE_STARTED:
            self._state = self.STATE_CLOSED
            self._client.remove_listener(self._session_watcher)
            try:
                # We must invoke on_deleted outside background queue because:
                # 1. The background task has been stopped.
                # 2. The on_deleted on closed tree does not communicate with
                #    ZooKeeper actually.
                self._root.on_deleted()
            except Exception as e:
                logger.debug('processing error on close(): %r', e)

    def listen(self, listener):
        """Registers a function to listen the cache events.

        The cache events are changes of local data. They are delivered from
        watching notifications in ZooKeeper session.

        This method can be use as a decorator.

        :param listener: A callable object which accepting a
                         :class:`~kazoo.recipe.cache.TreeEvent` instance as
                         its argument.
        """
        self._event_listeners.append(listener)
        return listener

    def get_data(self, path, default=None):
        """Gets data of a node from cache.

        :param path: The absolute path string.
        :param default: The default value which will be returned if the node
                        does not exist.
        :raises ValueError: If the path is outside of this subtree.
        :returns: A :class:`~kazoo.recipe.cache.NodeData` instance.
        """
        node = self._find_node(path)
        return default if node is None else node._data

    def get_children(self, path, default=None):
        """Gets node children list from in-memory snapshot.

        :param path: The absolute path string.
        :param default: The default value which will be returned if the node
                        does not exist.
        :raises ValueError: If the path is outside of this subtree.
        :returns: The :class:`frozenset` which including children names.
        """
        node = self._find_node(path)
        return default if node is None else frozenset(node._children)

    def _find_node(self, path):
        if not path.startswith(self._root._path):
            raise ValueError('outside of tree')
        stripped_path = path[len(self._root._path):].strip('/')
        split_path = (p for p in stripped_path.split('/') if p)
        current_node = self._root
        for node_name in split_path:
            if node_name not in current_node._children:
                return
            current_node = current_node._children[node_name]
        return current_node

    def _publish_event(self, event_type, event_data=None):
        event = TreeEvent.make(event_type, event_data)
        if self._state != self.STATE_CLOSED:
            self._in_background(self._do_publish_event, event)

    def _do_publish_event(self, event):
        for listener in self._event_listeners:
            try:
                listener(event)
            except Exception as e:
                logger.debug('processing error on _do_publish_event(): %s; event: %s', e, event)

    def _in_background(self, func, *args, **kwargs):
        self._task_queue.put((func, args, kwargs))

    def _initial_in_background(self, func, *args, **kwargs):
        kwargs['initial'] = True
        self._initial_task_queue.put((func, args, kwargs))

    def _construct_queue_handler_method(self, queue, lock):
        def handler():
            i = 0
            while 1:
                try:
                    cb = queue.get()
                    func, args, kwargs = cb
                    with lock:
                        func(*args, **kwargs)
                    # release before possible idle
                    del cb, func, args, kwargs
                except Exception as e:
                    logger.debug('processing error on handler(): %r', e)
                else:
                    if i == 10000:
                        # Let's idle from time to time, because a long queue can lock the execution context.
                        # Let's not use modular division to save a little bit of CPU.
                        gevent.idle()
                        i = 0
                    i += 1

        return handler

    def _session_watcher(self, state):
        if state == KazooState.SUSPENDED:
            self._publish_event(TreeEvent.CONNECTION_SUSPENDED)
        elif state == KazooState.CONNECTED:
            # The session watcher should not be blocked
            self._initial_in_background(self._root.on_reconnected)
            self._publish_event(TreeEvent.CONNECTION_RECONNECTED)
        elif state == KazooState.LOST:
            self._is_initialized = False
            self._publish_event(TreeEvent.CONNECTION_LOST)


class TreeNode(object):
    """The tree node record.

    :param tree: A :class:`~kazoo.recipe.cache.TreeCache` instance.
    :param path: The path of current node.
    :param parent: The parent node reference. ``None`` for root node.
    :param Structure structure:
    """

    __slots__ = ('_tree', '_path', '_prefixed_path', '_parent', '_structure', '_codec', '_children',
                 '_state', '_data', '_client_get', '_client_get_children', '_client_exists')

    STATE_PENDING = 0
    STATE_LIVE = 1
    STATE_DEAD = 2

    def __init__(self, tree, path, parent, structure=None):
        self._tree = tree
        self._path = path.rstrip('/')
        self._prefixed_path = _prefix_root(self._tree._client.chroot, self._path)
        self._parent = parent
        self._structure = structure
        self._codec = structure.get_codec() if structure else None
        self._children = {}
        self._state = self.STATE_PENDING
        self._data = None
        self._client_get = self._construct_method('get')
        self._client_get_children = self._construct_method('get_children')
        self._client_exists = self._construct_method('exists')

    @classmethod
    def make_root(cls, tree, path, structure=None):
        return cls(tree, path, None, structure=structure)

    def on_reconnected(self, initial=False):
        self._refresh(initial=initial)
        for child in self._children.values():
            child.on_reconnected(initial=initial)

    def on_created(self, initial=False):
        self._refresh(initial=initial)

    def on_deleted(self):
        old_children, self._children = self._children, {}
        old_data, self._data = self._data, None

        for old_child in old_children.values():
            old_child.on_deleted()

        if self._tree._state == self._tree.STATE_CLOSED:
            self._reset_watchers()
            return

        old_state, self._state = self._state, self.STATE_DEAD
        if old_state == self.STATE_LIVE:
            self._publish_event(TreeEvent.NODE_REMOVED, old_data)

        if self._parent is None:
            self._client_exists()  # root node
        else:
            child = self._path[len(self._parent._path):].strip('/')
            if self._parent._children.get(child) is self:
                del self._parent._children[child]
                self._reset_watchers()

    def _publish_event(self, *args, **kwargs):
        return self._tree._publish_event(*args, **kwargs)

    def _reset_watchers(self):
        client = self._tree._client
        if self._prefixed_path in client._data_watchers:
            client._data_watchers[self._prefixed_path].discard(self._process_watch)
        if self._prefixed_path in client._child_watchers:
            client._child_watchers[self._prefixed_path].discard(self._process_watch)

    def _refresh(self, initial=False):
        self._refresh_data(initial=initial)
        self._refresh_children(initial=initial)

    def _refresh_data(self, initial=False):
        self._client_get(initial=initial)

    def _refresh_children(self, initial=False):
        if self._structure and self._structure.is_leaf():
            return
        self._client_get_children(initial=initial)

    def _construct_method(self, method_name):
        callback = functools.partial(self._tree._in_background, self._process_result, method_name, self._path)
        initial_callback = functools.partial(self._tree._initial_in_background, self._process_result, method_name,
                                             self._path)
        method = getattr(self._tree._client, method_name + '_async')

        def f(initial=False):
            if initial:
                self._tree._initial_outstanding_ops += 1
            else:
                self._tree._outstanding_ops += 1

            method(self._path, watch=self._process_watch).rawlink(initial_callback if initial else callback)

        return f

    def _process_watch(self, watched_event):
        try:
            if watched_event.type == EventType.CREATED:
                assert self._parent is None, 'unexpected CREATED on non-root'
                self.on_created()
            elif watched_event.type == EventType.DELETED:
                self.on_deleted()
            elif watched_event.type == EventType.CHANGED:
                self._refresh_data()
            elif watched_event.type == EventType.CHILD:
                self._refresh_children()
        except Exception as e:
            logger.debug('processing error on _process_watch(): %r', e)

    def _process_result(self, method_name, path, result, initial=False):
        if method_name == 'exists':
            assert self._parent is None, 'unexpected EXISTS on non-root'
            # The result will be `None` if the node doesn't exist.
            if result.successful() and result.get() is not None:
                if self._state == self.STATE_DEAD:
                    self._state = self.STATE_PENDING
                self.on_created(initial=initial)
        elif method_name == 'get_children':
            if result.successful():
                children = result.get()
                for child in children:
                    child_structure = None
                    if self._structure:
                        child_structure = self._structure.get_child(child)
                        if child_structure is None:
                            continue
                    full_path = os.path.join(path, child)
                    if child not in self._children:
                        node = TreeNode(self._tree, full_path, self, structure=child_structure)
                        self._children[child] = node
                        node.on_created(initial=initial)
            elif isinstance(result.exception, NoNodeError):
                self.on_deleted()
        elif method_name == 'get':
            if result.successful():
                data, stat = result.get()
                if self._codec:
                    data = self._codec.decode(data, stat=stat)
                old_data, self._data = self._data, NodeData.make(path, data, stat)
                old_state, self._state = self._state, self.STATE_LIVE
                if old_state == self.STATE_LIVE:
                    if old_data is None or old_data.stat.mzxid != stat.mzxid:
                        self._publish_event(TreeEvent.NODE_UPDATED, self._data)
                else:
                    self._publish_event(TreeEvent.NODE_ADDED, self._data)
            elif isinstance(result.exception, NoNodeError):
                self.on_deleted()
        else:  # pragma: no cover
            logger.warning('unknown operation %s', method_name)
            if initial:
                self._tree._initial_outstanding_ops -= 1
            else:
                self._tree._outstanding_ops -= 1
            return

        if initial:
            self._tree._initial_outstanding_ops -= 1
        else:
            self._tree._outstanding_ops -= 1
        if (self._tree._initial_outstanding_ops == 0 and
                len(self._tree._initial_task_queue) == 0 and
                not self._tree._is_initialized):
            self._tree._is_initialized = True
            self._publish_event(TreeEvent.INITIALIZED)


class TreeEvent(tuple):
    """The immutable event tuple of cache."""

    NODE_ADDED = 0
    NODE_UPDATED = 1
    NODE_REMOVED = 2
    CONNECTION_SUSPENDED = 3
    CONNECTION_RECONNECTED = 4
    CONNECTION_LOST = 5
    INITIALIZED = 6

    #: An enumerate integer to indicate event type.
    event_type = property(operator.itemgetter(0))

    #: A :class:`~kazoo.recipe.cache.NodeData` instance.
    event_data = property(operator.itemgetter(1))

    @classmethod
    def make(cls, event_type, event_data):
        """Creates a new TreeEvent tuple.

        :returns: A :class:`~kazoo.recipe.cache.TreeEvent` instance.
        """
        return cls((event_type, event_data))


class NodeData(tuple):
    """The immutable node data tuple of cache."""

    #: The absolute path string of current node.
    path = property(operator.itemgetter(0))

    #: The bytes data of current node.
    data = property(operator.itemgetter(1))

    #: The stat information of current node.
    stat = property(operator.itemgetter(2))

    @classmethod
    def make(cls, path, data, stat):
        """Creates a new NodeData tuple.

        :returns: A :class:`~kazoo.recipe.cache.NodeData` instance.
        """
        return cls((path, data, stat))


class DataNode(Structure):
    def __init__(self, codec=None):
        """
        :type codec: Optional[awacs.lib.storage.Codec]
        """
        self._codec = codec

    def __repr__(self):
        return u'DataNode(codec={})'.format(self._codec.__name__ if self._codec is not None else None)

    def __eq__(self, other):
        return six.text_type(self) == six.text_type(other)

    def get_codec(self):
        return self._codec

    def get_child(self, path):
        return None

    def is_leaf(self):
        return True


class SkippedNode(Structure):
    def __init__(self, nested):
        """
        :type nested: treecache.Structure
        """
        self._nested = nested

    def __repr__(self):
        return u'SkippedNode({})'.format(self._nested)

    def __eq__(self, other):
        return self._nested == other._nested

    def get_codec(self):
        return None

    def get_child(self, path):
        return self._nested

    def is_leaf(self):
        return False


class RootStructure(Structure):
    def __init__(self, nested):
        """
        :type nested: dict[str, treecache.Structure]
        """
        self._nested = nested

    def __repr__(self):
        return u'RootStructure({})'.format(sorted(self._nested.items()))

    def __eq__(self, other):
        return self._nested == other._nested

    def get_codec(self):
        return None

    def get_child(self, path):
        return self._nested.get(path)

    def is_leaf(self):
        return False
