from __future__ import absolute_import

import io
import socket
import logging

import gevent.queue
import kazoo.loggingsupport
import kazoo.python2atexit
from gevent.hub import getcurrent
from gevent.event import AsyncResult as GeventAsyncResult
from infra.swatlib.zk.client import ZookeeperClient as _ZookeeperClient
from kazoo.handlers.gevent import SequentialGeventHandler as BaseSequentialGeventHandler, _STOP
from kazoo.interfaces import IAsyncResult


class IZookeeperClient(object):
    pass


class ZookeeperInstanceState(object):
    def __init__(self, host, port, is_ok, monitoring_result=None):
        self.host = host
        self.port = port
        self.is_ok = is_ok
        self.monitoring_result = monitoring_result

    def __str__(self):
        return u"ZookeeperInstanceState({})".format(self.__dict__)

    __repr__ = __str__


class ZookeeperClusterState(object):
    def __init__(self):
        self.instances = []

    def __str__(self):
        return u"ZookeeperClusterState(instances={})".format(self.instances)


class InstanceMonitoringResult(object):
    def __init__(self):
        self.zk_version = None
        self.avg_latency = None
        self.zk_max_latency = None
        self.zk_min_latency = None
        self.zk_packets_received = None
        self.zk_packets_sent = None
        self.zk_outstanding_requests = None
        self.zk_server_state = None
        self.zk_znode_count = None
        self.zk_watch_count = None
        self.zk_ephemerals_count = None
        self.zk_approximate_data_size = None
        self.zk_followers = None  # only exposed by the Leader
        self.zk_synced_followers = None  # only exposed by the Leader
        self.zk_pending_syncs = None  # only exposed by the Leader
        self.zk_open_file_descriptor_count = None  # only available on Unix platforms
        self.zk_max_file_descriptor_count = None  # only available on Unix platforms


class ZookeeperClient(IZookeeperClient, _ZookeeperClient):
    @staticmethod
    def _parse_mntr(data):
        """
        Parse mntr command output. It looks like this:
        zk_version  3.4.0
        zk_avg_latency  0
        zk_max_latency  0
        zk_min_latency  0
        zk_packets_received 70
        zk_packets_sent 69
        zk_outstanding_requests 0
        zk_server_state leader
        zk_znode_count   4
        zk_watch_count  0
        zk_ephemerals_count 0
        zk_approximate_data_size    27
        zk_followers    4                   - only exposed by the Leader
        zk_synced_followers 4               - only exposed by the Leader
        zk_pending_syncs    0               - only exposed by the Leader
        zk_open_file_descriptor_count 23    - only available on Unix platforms
        zk_max_file_descriptor_count 1024   - only available on Unix platforms
        """
        result = InstanceMonitoringResult()
        for line in io.StringIO(data):
            parts = line.rstrip().split('\t', 1)
            if len(parts) != 2:
                continue
            key, value = parts
            setattr(result, key, value)
        return result

    def __init__(self, cfg, identifier=None, handler=None, write_lock_timeout=None, metrics_registry=None):
        if handler is None:
            handler = SequentialGeventHandler()
        super(ZookeeperClient, self).__init__(cfg, identifier=identifier, handler=handler,
                                              metrics_registry=metrics_registry)
        if cfg.get(u'log_debug', False):
            self.client.logger.setLevel(kazoo.loggingsupport.BLATHER)
        if write_lock_timeout is not None:
            self.client.write_lock_timeout = write_lock_timeout

    def execute_4letter_command(self, addr, command, timeout=10):
        def recvall(s):
            """Receive from socket until connection is closed"""
            buf = io.BytesIO()
            while 1:
                try:
                    if not buf.write(s.recv(8192)):
                        break
                except EnvironmentError:
                    break
            return buf.getvalue()

        sock = None
        try:
            sock = self.client.handler.create_connection(addr, timeout=timeout)
            sock.sendall(command)
            return recvall(sock).decode(u'utf-8', u'replace')
        finally:
            if sock is not None:
                sock.close()

    def get_cluster_state(self):
        cluster = ZookeeperClusterState()
        for addr in self.client.hosts:
            is_ok = False
            try:
                is_ok = self.execute_4letter_command(addr, u'ruok') == u'imok'
            except socket.error:
                pass
            host, port = addr
            monitoring_result = None
            if is_ok:  # no need to ask mntr if "are you ok" failed
                try:
                    data = self.execute_4letter_command(addr, u'mntr')
                except socket.error:
                    pass
                else:
                    monitoring_result = self._parse_mntr(data)
            cluster.instances.append(ZookeeperInstanceState(host, port, is_ok, monitoring_result))
        return cluster


class OffloadedCallback(object):
    """
    A proxy to a function.
    Offloads all calls to the proxied function to the handler's completion queue.
    """
    __slots__ = ('handler', 'f')

    def __init__(self, handler, f):
        """
        :type handler: SequentialGeventHandler
        :type f: callable
        """
        self.handler = handler
        self.f = f

    def __call__(self, *args, **kwargs):
        return self.handler.completion_queue.put(lambda: self.f(*args, **kwargs))

    def __hash__(self):
        return hash(self.f)

    def __eq__(self, other):
        if isinstance(other, OffloadedCallback):
            return self.f == other.f
        else:
            return self.f == other


class AsyncResult(GeventAsyncResult, IAsyncResult):
    """
    AsyncResult implementation that offloads linked calls to its handler's completion
    queue instead of running them in the hub greenlet. The only exception are
    `Greenlet.switch` calls that still have to be executed in the hub greenlet,
    as usual rawlink-ed callback.
    """

    def __init__(self, handler):
        self._handler = handler
        GeventAsyncResult.__init__(self)

    def rawlink(self, callback):
        if callback != getcurrent().switch:
            callback = OffloadedCallback(self._handler, callback)
        return GeventAsyncResult.rawlink(self, callback)


class SequentialGeventHandler(BaseSequentialGeventHandler):
    async_result_impl = AsyncResult

    def __init__(self):
        BaseSequentialGeventHandler.__init__(self)
        self._callback_queue_worker = None
        self._completion_queue_worker = None
        self._cb_processed_while_stopped = 0
        self._log = logging.getLogger(__name__)

    def _create_greenlet_worker(self, queue):
        def greenlet_worker():
            while True:
                try:
                    func = queue.get()
                    if func is _STOP:
                        break
                    if not self._running:
                        self._cb_processed_while_stopped += 1
                    func()
                except gevent.queue.Empty:
                    continue
                except Exception as exc:
                    self._log.warning(u"Exception in worker greenlet")
                    self._log.exception(exc)

        return gevent.spawn(greenlet_worker)

    def start(self):
        """Start the greenlet workers."""
        with self._state_change:
            if self._running:
                return

            if self._callback_queue_worker is None:
                self._callback_queue_worker = self._create_greenlet_worker(self.callback_queue)
            if self._completion_queue_worker is None:
                self._completion_queue_worker = self._create_greenlet_worker(self.completion_queue)
            self._running = True
            kazoo.python2atexit.register(self.close)

    def stop(self):
        """Stop the greenlet workers and empty all queues."""
        with self._state_change:
            if not self._running:
                return

            self._running = False
            self.callback_queue.put(_STOP)
            self._callback_queue_worker.get()
            self._callback_queue_worker = None
            self.callback_queue = self.queue_impl()

    def close(self):
        self.stop()
        with self._state_change:
            if self._completion_queue_worker is not None:
                self.completion_queue.put(_STOP)
                self._completion_queue_worker.get()
                self._completion_queue_worker = None
                self.completion_queue = self.queue_impl()
        lvl = logging.WARN if self._cb_processed_while_stopped > 0 else logging.DEBUG
        self._log.log(lvl, u'number of callbacks processed while stopped: %d', self._cb_processed_while_stopped)
        kazoo.python2atexit.unregister(self.close)
