package main

import (
	"flag"
	"os"

	"log"

	"github.com/golang/protobuf/proto"
	"go.uber.org/zap"

	"a.yandex-team.ru/infra/tcp-sampler/pkg/hbfutil"
	"a.yandex-team.ru/infra/tcp-sampler/pkg/iputil"
	"a.yandex-team.ru/infra/tcp-sampler/pkg/lock"
	"a.yandex-team.ru/infra/tcp-sampler/pkg/portoutil"
	"a.yandex-team.ru/infra/tcp-sampler/pkg/samples"
	"a.yandex-team.ru/infra/tcp-sampler/pkg/valve"
	"a.yandex-team.ru/infra/tcp-sampler/pkg/yasm"
	pb "a.yandex-team.ru/mds/valve/proto"
)

const (
	hbfLogPath      = "/var/log/yandex-hbf-agent/yandex-hbf-agent.log"
	lockPath        = "/var/run/tcp-sampler.lock"
	dumpPathDefault = "/tmp/tcp-sampler.message.dump"
)

var hbfRulesPaths = []string{"/usr/share/yandex-hbf-agent/rules.d", "/etc/yandex-hbf-agent/rules.d"}

func yasmPush(r *yasm.Req) {
	if err := r.Push(yasm.URL); err != nil {
		zap.S().Warn(err)
	} else {
		zap.S().Info("Yasm Push OK")
	}
}

func main() {

	hbfurl := flag.String("hbfurl", hbfutil.HBFURL, "Set url to get hbf url from")
	disabledPush := flag.Bool("nopush", false, "Do not push to valve")
	dump := flag.Bool("nodump", false, "Dump result proto message to file")
	debug := flag.Bool("debug", false, "Debug level")
	dumpPath := flag.String("dump-path", dumpPathDefault, "Path to file where to dump proto")
	prevDump := flag.String("prev-dump", dumpPathDefault, "Path to previous dump file")
	flag.Parse()

	var logger *zap.Logger
	var err error

	if *debug {
		logger, err = zap.NewDevelopment()
		if err != nil {
			log.Fatalf("Can not init logger %v", err)
		}
	} else {
		logger, err = zap.NewProduction()
		if err != nil {
			log.Fatalf("Can not init logger %v", err)
		}
	}
	defer logger.Sync()
	undo := zap.ReplaceGlobals(logger)
	defer undo()

	l, err := lock.Lock(lockPath)
	if err != nil {
		zap.S().Fatal(err)
	}
	defer l.Unlock()

	tags := make(yasm.TagsMap)
	tags["itype"] = "rtcsys"
	tags["prj"] = "tcp-sampler"
	ttl := uint(13 * 60 * 60)

	sigPushOK := yasm.Signal{
		Name: "samples_push_ok_ahhh",
	}
	sigPushFail := yasm.Signal{
		Name: "samples_push_fail_ahhh",
	}
	sigValidateFail := yasm.Signal{
		Name: "samples_validate_fail_ahhh",
	}

	netSample := &pb.NetSample{}

	hostname, err := os.Hostname()
	if err != nil {
		zap.S().Fatal(err)
	} else {
		netSample.Fqdn = hostname
	}

	url, err := hbfutil.GetLastHBFURL(hbfurl)
	if err != nil {
		zap.S().Debugf("%v", err)
	} else {
		netSample.HbfUrl = url
	}

	notInherited, err := portoutil.GetNotInherited()
	if err != nil {
		log.Fatal(err)
	}

	neighs, err := iputil.GetNeighReach()
	if err != nil {
		zap.S().Debugf("Neigh err: %v", err)
	}
	nsFull := samples.FormNetSampleNSMap(notInherited, neighs)

	oldSample, err := samples.ReadOldSample(*prevDump)
	if err != nil {
		netSample.Ns = nsFull
	} else {
		nsFiltered := samples.Filter(nsFull, oldSample)
		netSample.Ns = nsFiltered
	}

	lm, err := hbfutil.GetLocalRules(hbfRulesPaths)
	if err != nil {
		zap.S().Debug(err)
	} else {
		netSample.LocalHbfRules = lm
	}

	if !*dump {
		fs := &pb.NetSample{Ns: nsFull, Fqdn: hostname, HbfUrl: url, LocalHbfRules: lm}
		zap.S().Debugf("Protos are equal: %v", proto.Equal(fs, netSample))
		err := samples.WriteProtoMessageToFile(*dumpPath, fs)
		if err != nil {
			zap.S().Debugf("Got err: %v", err)
		}
	}
	err = netSample.Validate()
	if err != nil {
		sigValidateFail.Val = 1
		r := yasm.NewReq(ttl, tags, sigPushOK, sigPushFail, sigValidateFail)
		yasmPush(r)
		zap.S().Fatal(err)
	}

	if !*disabledPush {
		c := valve.NewClient(valve.APIURL)
		_, err := c.NewSample(netSample)
		if err != nil {
			zap.S().Debug(err)
			sigPushFail.Val = 1
			r := yasm.NewReq(ttl, tags, sigPushOK, sigPushFail, sigValidateFail)
			yasmPush(r)
		} else {
			zap.S().Info("Pushed OK")
			sigPushOK.Val = 1
			r := yasm.NewReq(ttl, tags, sigPushOK, sigPushFail, sigValidateFail)
			yasmPush(r)
		}
	}
}
