package iputil

import (
	"encoding/binary"
	"encoding/hex"
	"errors"
	"net"

	"github.com/vishvananda/netlink"
	"go.uber.org/zap"
	"golang.org/x/sys/unix"
)

// ConvertIP6FromSS used to convert /proc/net/tcp6 output of ip6s to proper ip6 format.
func ConvertIP6FromSS(buf []byte) (net.IP, error) {
	tmp := [16]byte{}
	n, err := hex.Decode(tmp[:], buf)
	if err != nil {
		return nil, err
	}
	if n == 4 {
		return net.IP{tmp[3], tmp[2], tmp[1], tmp[0]}, nil
	} else if n == 16 {
		t := make(net.IP, 16)
		binary.BigEndian.PutUint32(t[0:4], binary.LittleEndian.Uint32(tmp[0:4]))
		binary.BigEndian.PutUint32(t[4:8], binary.LittleEndian.Uint32(tmp[4:8]))
		binary.BigEndian.PutUint32(t[8:12], binary.LittleEndian.Uint32(tmp[8:12]))
		binary.BigEndian.PutUint32(t[12:16], binary.LittleEndian.Uint32(tmp[12:16]))
		return t, nil
	}
	return nil, errors.New("unexpected number of decoded bytes")
}

func GetHostIPs() (map[string]struct{}, error) {
	hostIPsSlice, err := net.InterfaceAddrs()
	if err != nil {
		return nil, err
	}
	zap.S().Debugf("HostIPsSlice: %#v", hostIPsSlice)
	hostIPNetSet := make(map[string]struct{})
	for _, a := range hostIPsSlice {
		if n, ok := a.(*net.IPNet); ok {
			hostIPNetSet[n.IP.String()] = struct{}{}
		}
	}
	zap.S().Debugf("HostIPsSet: %#v", hostIPNetSet)
	return hostIPNetSet, nil
}

func GetNeighReach() (map[string]string, error) {
	m := make(map[string]string)
	nL, err := netlink.NeighList(0, unix.AF_INET6)
	if err != nil {
		return nil, err
	}
	for _, n := range nL {
		if n.State == netlink.NUD_REACHABLE {
			idx := n.LinkIndex
			iname, err := net.InterfaceByIndex(idx)
			if err != nil {
				continue
			}
			m[n.IP.String()] = iname.Name
		}
	}
	zap.S().Debugf("NeighMap: %#v\n", m)
	return m, nil
}
