package sockdiag

import (
	"bytes"
	"io"
)

// TCP States
const (
	TCPNone = iota
	TCPEstablished
	TCPSynSent
	TCPSynRecv
	TCPFinWait1
	TCPFinWait2
	TCPTimeWait
	TCPClose
	TCPCloseWait
	TCPLastAck
	TCPListen
	TCPClosing
	TCPNewSynRec
	TCPMaxStates
)

type TCPInfo struct {
	State                  uint8
	CaState                uint8
	Retransmits            uint8
	Probes                 uint8
	Backoff                uint8
	Options                uint8
	SndWscale              uint8 // no uint4
	RcvWscale              uint8
	DeliveryRateAppLimited uint8
	FastopenClientFail     uint8
	Rto                    uint32
	Ato                    uint32
	SndMss                 uint32
	RcvMss                 uint32
	Unacked                uint32
	Sacked                 uint32
	Lost                   uint32
	Retrans                uint32
	Fackets                uint32
	LastDataSent           uint32
	LastAckSent            uint32
	LastDataRecv           uint32
	LastAckRecv            uint32
	Pmtu                   uint32
	RcvSsthresh            uint32
	Rtt                    uint32
	Rttvar                 uint32
	SndSsthresh            uint32
	SndCwnd                uint32
	Advmss                 uint32
	Reordering             uint32
	RcvRtt                 uint32
	RcvSpace               uint32
	TotalRetrans           uint32
	PacingRate             uint64
	MaxPacingRate          uint64
	BytesAcked             uint64 /* RFC4898 tcpEStatsAppHCThruOctetsAcked */
	BytesReceived          uint64 /* RFC4898 tcpEStatsAppHCThruOctetsReceived */
	SegsOut                uint32 /* RFC4898 tcpEStatsPerfSegsOut */
	SegsIn                 uint32 /* RFC4898 tcpEStatsPerfSegsIn */
	NotsentBytes           uint32
	MinRtt                 uint32
	DataSegsIn             uint32 /* RFC4898 tcpEStatsDataSegsIn */
	DataSegsOut            uint32 /* RFC4898 tcpEStatsDataSegsOut */
	DeliveryRate           uint64
	BusyTime               uint64 /* Time (usec) busy sending data */
	RwndLimited            uint64 /* Time (usec) limited by receive window */
	SndbufLimited          uint64 /* Time (usec) limited by send buffer */
	Delivered              uint32
	DeliveredCe            uint32
	BytesSent              uint64 /* RFC4898 tcpEStatsPerfHCDataOctetsOut */
	BytesRetrans           uint64 /* RFC4898 tcpEStatsPerfOctetsRetrans */
	DsackDups              uint32 /* RFC4898 tcpEStatsStackDSACKDups */
	ReordSeen              uint32 /* reordering events seen */
	RcvOoopack             uint32 /* Out-of-order packets received */
	SndWnd                 uint32 /* peer's advertised receive window after * scaling (bytes) */
}

func checkDeserErr(err error) error {
	if err == io.EOF {
		return nil
	}
	return err
}

func (t *TCPInfo) deserialize(b []byte) error {
	var err error
	rb := bytes.NewBuffer(b)

	t.State, err = rb.ReadByte()
	if err != nil {
		return checkDeserErr(err)
	}

	t.CaState, err = rb.ReadByte()
	if err != nil {
		return checkDeserErr(err)
	}

	t.Retransmits, err = rb.ReadByte()
	if err != nil {
		return checkDeserErr(err)
	}

	t.Probes, err = rb.ReadByte()
	if err != nil {
		return checkDeserErr(err)
	}

	t.Backoff, err = rb.ReadByte()
	if err != nil {
		return checkDeserErr(err)
	}
	t.Options, err = rb.ReadByte()
	if err != nil {
		return checkDeserErr(err)
	}

	scales, err := rb.ReadByte()
	if err != nil {
		return checkDeserErr(err)
	}
	t.SndWscale = scales >> 4  // first 4 bits
	t.RcvWscale = scales & 0xf // last 4 bits

	rateLimAndFastOpen, err := rb.ReadByte()
	if err != nil {
		return checkDeserErr(err)
	}
	t.DeliveryRateAppLimited = rateLimAndFastOpen >> 7 // get first bit
	t.FastopenClientFail = rateLimAndFastOpen >> 5 & 3 // get next two bits

	next := rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.Rto = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.Ato = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.SndMss = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.RcvMss = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.Unacked = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.Sacked = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.Lost = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.Retrans = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.Fackets = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.LastDataSent = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.LastAckSent = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.LastDataRecv = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.LastAckRecv = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.Pmtu = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.RcvSsthresh = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.Rtt = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.Rttvar = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.SndSsthresh = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.SndCwnd = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.Advmss = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.Reordering = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.RcvRtt = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.RcvSpace = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.TotalRetrans = native.Uint32(next)

	next = rb.Next(8)
	if len(next) == 0 {
		return nil
	}
	t.PacingRate = native.Uint64(next)

	next = rb.Next(8)
	if len(next) == 0 {
		return nil
	}
	t.MaxPacingRate = native.Uint64(next)

	next = rb.Next(8)
	if len(next) == 0 {
		return nil
	}
	t.BytesAcked = native.Uint64(next)

	next = rb.Next(8)
	if len(next) == 0 {
		return nil
	}
	t.BytesReceived = native.Uint64(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.SegsOut = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.SegsIn = native.Uint32(next)
	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.NotsentBytes = native.Uint32(next)
	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.MinRtt = native.Uint32(next)
	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.DataSegsIn = native.Uint32(next)
	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.DataSegsOut = native.Uint32(next)

	next = rb.Next(8)
	if len(next) == 0 {
		return nil
	}
	t.DeliveryRate = native.Uint64(next)

	next = rb.Next(8)
	if len(next) == 0 {
		return nil
	}
	t.BusyTime = native.Uint64(next)

	next = rb.Next(8)
	if len(next) == 0 {
		return nil
	}
	t.RwndLimited = native.Uint64(next)

	next = rb.Next(8)
	if len(next) == 0 {
		return nil
	}
	t.SndbufLimited = native.Uint64(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.Delivered = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.DeliveredCe = native.Uint32(next)

	next = rb.Next(8)
	if len(next) == 0 {
		return nil
	}
	t.BytesSent = native.Uint64(next)

	next = rb.Next(8)
	if len(next) == 0 {
		return nil
	}
	t.BytesRetrans = native.Uint64(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.DsackDups = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.ReordSeen = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.RcvOoopack = native.Uint32(next)

	next = rb.Next(4)
	if len(next) == 0 {
		return nil
	}
	t.SndWnd = native.Uint32(next)
	return nil
}
