package sysctl

import (
	"io/ioutil"
	"path/filepath"
	"regexp"
	"strconv"
	"strings"

	"go.uber.org/zap"
)

const (
	sysctlBasePath       = "/proc/sys/"
	defMinLocalPortRange = 32768
	defMaxLocalPortRange = 65535
)

var localPortRangeRE = regexp.MustCompile(`(?m)^(?P<min>[0-9]+)\s+(?P<max>[0-9]+)$`)

func readFileAsString(s string) (string, error) {
	b, err := ioutil.ReadFile(s)
	if err != nil {
		return "", err
	}
	return string(b), nil
}

// sysctlPath is a simple way to get path to sysctl
// if you want to reuse it, it is better to change it, like it is done in netonfig
func sysctlPath(s string) string {
	return filepath.Join(sysctlBasePath, strings.Replace(s, ".", "/", -1))
}

func Get(s string) (string, error) {
	return readFileAsString(sysctlPath(s))
}

func GetLocalPortRange() (uint32, uint32) {
	lp, err := Get("net.ipv4.ip_local_port_range")
	if err != nil {
		zap.S().Debugf("Could not get local_port_range: %v", err)
		zap.S().Debug("Returning default values\n")
		return uint32(defMinLocalPortRange), uint32(defMaxLocalPortRange)
	}
	zap.S().Debugf("Got: %#v", lp)
	sl := localPortRangeRE.FindAllStringSubmatch(lp, -1)
	zap.S().Debugf("SL: %#v", sl)

	if len(sl) == 1 {
		min, err := strconv.ParseUint(sl[0][1], 10, 64)
		if err == nil {
			max, err := strconv.ParseUint(sl[0][2], 10, 64)
			if err == nil {
				return uint32(min), uint32(max)
			}
		}
	}
	zap.S().Debugf("Stranger things in local_port_range: %v", sl)
	zap.S().Debug("Returning default values")
	return uint32(defMinLocalPortRange), uint32(defMaxLocalPortRange)
}
