package valve

import (
	"bytes"
	"compress/gzip"
	"fmt"
	"io/ioutil"
	"net/http"
	"time"

	pb "a.yandex-team.ru/mds/valve/proto"
	"github.com/golang/protobuf/proto"
)

const (
	APIURL                = "http://valve.yandex.net/ValveServer/"
	acceptHeader          = "Accept"
	contentTypeHeader     = "Content-Type"
	contentEncodingHeader = "Content-Encoding"
	contentEncodingValue  = "gzip"
	userAgentHeader       = "User-Agent"
	userAgentValue        = "tcp-sampler-client"
	protobufHeader        = "application/x-protobuf"
)

type Client struct {
	url        string
	headers    http.Header
	httpClient http.Client
}

func NewClient(url string) *Client {
	return &Client{
		headers: http.Header{
			acceptHeader:          []string{protobufHeader},
			contentTypeHeader:     []string{protobufHeader},
			contentEncodingHeader: []string{contentEncodingValue},
			userAgentHeader:       []string{userAgentValue},
		},
		url: url,
		httpClient: http.Client{
			Timeout: 5 * time.Second,
		},
	}
}

func (c *Client) do(method string, req proto.Message) (int, error) {
	buf, err := proto.Marshal(req)
	if err != nil {
		return 0, err
	}

	var gziped bytes.Buffer
	gzw := gzip.NewWriter(&gziped)
	_, err = gzw.Write(buf)
	if err != nil {
		return 0, err
	}
	err = gzw.Flush()
	if err != nil {
		return 0, err
	}
	err = gzw.Close()
	if err != nil {
		return 0, err
	}

	httpReq, err := http.NewRequest("POST", c.url+method+"/", &gziped)
	if err != nil {
		return 0, err
	}
	httpReq.Header = c.headers
	httpResp, err := c.httpClient.Do(httpReq)
	if err != nil {
		return 0, err
	}
	defer func() {
		_ = httpResp.Body.Close()
	}()
	if httpResp.StatusCode != 200 {
		return 0, fmt.Errorf("bad response code: %d, resp_headers: %v", httpResp.StatusCode, httpResp.Header)
	}
	_, err = ioutil.ReadAll(httpResp.Body)
	if err != nil {
		return 0, fmt.Errorf("failed to read response: %s", err.Error())
	}
	return len(buf), nil
}

func (c *Client) NewSample(req *pb.NetSample) (int, error) {
	return c.do("NewSample", req)
}
