package awacs

import (
	"a.yandex-team.ru/infra/awacs/clients/go/awacs"
	awacspb "a.yandex-team.ru/infra/awacs/proto"
	"context"
	"errors"
	"google.golang.org/protobuf/types/known/fieldmaskpb"
	timestamppb "google.golang.org/protobuf/types/known/timestamppb"
	"time"
)

type PermanentError struct {
	s string
	e error
}

func (e *PermanentError) Error() string {
	return e.s
}

func (e *PermanentError) Unwrap() error {
	return e.e
}

func NewPermanentError(s string, e error) *PermanentError {
	return &PermanentError{s: s, e: e}
}

func isPermanent(err error) bool {
	_, ok := err.(*PermanentError)
	return ok
}

type client struct {
	permanentErrorChecks []PermanentErrorCheck
	c                    awacs.Client
}

type PermanentErrorCheck func(*awacs.APIError) bool
type Option func(*client)

func WithPermanentErrorCheck(check PermanentErrorCheck) Option {
	return func(c *client) {
		c.permanentErrorChecks = append(c.permanentErrorChecks, check)
	}
}

func NewClient(apiURL, token string, opts ...Option) *client {
	c := &client{
		c: awacs.NewClient(
			awacs.WithToken(token),
			awacs.WithAPIURL(apiURL),
		),
	}
	for _, opt := range opts {
		opt(c)
	}
	return c
}

func (c *client) maybeWrap(err error) error {
	var apiError *awacs.APIError
	if errors.As(err, &apiError) {
		for _, check := range c.permanentErrorChecks {
			if check(apiError) {
				return NewPermanentError(err.Error(), err) // TODO romanovich@: wrap, not cast to string?
			}
		}
	}
	return err
}

// Namespaces

func (c *client) ListSummaries(ctx context.Context) ([]*awacspb.ListNamespaceSummariesResponse_Summary, error) {
	req := &awacspb.ListNamespaceSummariesRequest{}
	rsp, err := c.c.ListNamespaceSummaries(ctx, req)
	if err != nil {
		return nil, c.maybeWrap(err)
	}
	return rsp.GetSummaries(), nil
}

// Balancers

func (c *client) GetBalancer(ctx context.Context, namespaceID, id string) (*awacspb.Balancer, error) {
	req := &awacspb.GetBalancerRequest{NamespaceId: namespaceID, Id: id}
	rsp, err := c.c.GetBalancer(ctx, req)
	if err != nil {
		return nil, c.maybeWrap(err)
	}
	return rsp.GetBalancer(), nil
}

func (c *client) GetBalancerState(ctx context.Context, namespaceID, id string) (*awacspb.BalancerState, error) {
	req := &awacspb.GetBalancerStateRequest{NamespaceId: namespaceID, Id: id}
	rsp, err := c.c.GetBalancerState(ctx, req)
	if err != nil {
		return nil, c.maybeWrap(err)
	}
	return rsp.GetState(), nil
}

func (c *client) ListNamespaces(ctx context.Context, fieldMask *fieldmaskpb.FieldMask) ([]*awacspb.Namespace, error) {
	req := &awacspb.ListNamespacesRequest{FieldMask: fieldMask}
	rsp, err := c.c.ListNamespaces(ctx, req)
	if err != nil {
		return nil, err
	}

	return rsp.Namespaces, nil
}

func (c *client) ListBalancersByNamespaceID(ctx context.Context, namespaceID string, m *fieldmaskpb.FieldMask) ([]*awacspb.Balancer, error) {
	req := &awacspb.ListBalancersRequest{NamespaceId: namespaceID, FieldMask: m}
	rsp, err := c.c.ListBalancers(ctx, req)
	if err != nil {
		return nil, c.maybeWrap(err)
	}
	return rsp.GetBalancers(), nil
}

func (c *client) ListL3BalancersByNamespaceID(ctx context.Context, namespaceID string, m *fieldmaskpb.FieldMask) ([]*awacspb.L3Balancer, error) {
	req := &awacspb.ListL3BalancersRequest{NamespaceId: namespaceID, FieldMask: m}
	rsp, err := c.c.ListL3Balancers(ctx, req)
	if err != nil {
		return nil, c.maybeWrap(err)
	}
	return rsp.GetL3Balancers(), nil
}

func (c *client) SetBalancerSpec(ctx context.Context, namespaceID, balancerID, version, comment string, spec *awacspb.BalancerSpec) (*awacspb.Balancer, error) {
	req := &awacspb.UpdateBalancerRequest{
		Meta: &awacspb.BalancerMeta{
			Id:          balancerID,
			NamespaceId: namespaceID,
			Version:     version,
			Comment:     comment,
		},
		Spec: spec,
	}
	rsp, err := c.c.UpdateBalancer(ctx, req)
	if err != nil {
		return nil, c.maybeWrap(err)
	}
	return rsp.GetBalancer(), nil
}

// Certificates

func (c *client) GetCertificate(ctx context.Context, namespaceID, id string) (*awacspb.Certificate, error) {
	req := &awacspb.GetCertificateRequest{NamespaceId: namespaceID, Id: id}
	rsp, err := c.c.GetCertificate(ctx, req)
	if err != nil {
		return nil, c.maybeWrap(err)
	}
	return rsp.GetCertificate(), nil
}

func (c *client) GetCertificateRenewal(ctx context.Context, namespaceID, id string) (*awacspb.CertificateRenewal, error) {
	req := &awacspb.GetCertificateRenewalRequest{NamespaceId: namespaceID, Id: id}
	rsp, err := c.c.GetCertificateRenewal(ctx, req)
	if err != nil {
		return nil, c.maybeWrap(err)
	}
	return rsp.GetCertificateRenewal(), nil
}

func (c *client) ListCertificateRenewals(ctx context.Context, req *awacspb.ListCertificateRenewalsRequest) ([]*awacspb.CertificateRenewal, error) {
	rsp, err := c.c.ListCertificateRenewals(ctx, req)
	if err != nil {
		return nil, c.maybeWrap(err)
	}
	return rsp.GetCertificateRenewals(), nil
}

func (c *client) SetCertificateRenewalTargetDiscoverability(ctx context.Context, namespaceID, id, version string, targetDiscoverability *awacspb.DiscoverabilityCondition) error {
	req := &awacspb.UnpauseCertificateRenewalRequest{
		NamespaceId:           namespaceID,
		Id:                    id,
		Version:               version,
		TargetDiscoverability: targetDiscoverability,
	}
	_, err := c.c.UnpauseCertificateRenewal(ctx, req)
	if err != nil {
		return c.maybeWrap(err)
	}
	return nil
}

func (c *client) SetCertificateMeta(ctx context.Context, namespaceID, id string, meta *awacspb.CertificateMeta) error {
	if namespaceID != meta.NamespaceId || id != meta.Id {
		return errors.New("`namespaceID` and `id` do not match `meta`")
	}
	req := &awacspb.UpdateCertificateRequest{
		Meta: meta,
	}
	_, err := c.c.UpdateCertificate(ctx, req)
	if err != nil {
		return c.maybeWrap(err)
	}
	return nil
}

func (c *client) PauseBalancer(ctx context.Context, namespaceID, balancerID string) error {
	req := &awacspb.UpdateBalancerTransportPausedRequest{
		Id:              balancerID,
		NamespaceId:     namespaceID,
		TransportPaused: &awacspb.PausedCondition{Value: true},
	}

	_, err := c.c.UpdateBalancerTransportPaused(ctx, req)
	if err != nil {
		return c.maybeWrap(err)
	}

	return nil
}

func (c *client) UnpauseBalancer(ctx context.Context, namespaceID, balancerID string) error {
	req := &awacspb.UpdateBalancerTransportPausedRequest{
		Id:              balancerID,
		NamespaceId:     namespaceID,
		TransportPaused: &awacspb.PausedCondition{Value: false},
	}

	_, err := c.c.UpdateBalancerTransportPaused(ctx, req)
	if err != nil {
		return c.maybeWrap(err)
	}

	return nil
}

func (c *client) ListBackends(ctx context.Context, namespaceID string) ([]*awacspb.Backend, error) {
	req := &awacspb.ListBackendsRequest{
		NamespaceId: namespaceID,
	}

	rsp, err := c.c.ListBackends(ctx, req)
	if err != nil {
		return nil, c.maybeWrap(err)
	}

	return rsp.Backends, nil
}

func (c *client) ListBackendRevisions(
	ctx context.Context,
	namespaceID string,
	backendID string,
	skip int32,
	limit int32) (*awacspb.ListBackendRevisionsResponse, error) {

	req := &awacspb.ListBackendRevisionsRequest{
		Id:          backendID,
		NamespaceId: namespaceID,
		Skip:        skip,
		Limit:       limit,
	}
	rsp, err := c.c.ListBackendRevisions(ctx, req)
	if err != nil {
		return nil, c.maybeWrap(err)
	}

	return rsp, nil
}

func (c *client) GetNamespaceAspectsSet(ctx context.Context, namespaceID string) (*awacspb.GetNamespaceAspectsSetResponse, error) {
	req := &awacspb.GetNamespaceAspectsSetRequest{
		Id: namespaceID,
	}

	rsp, err := c.c.GetNamespaceAspectsSet(ctx, req)
	if err != nil {
		return nil, c.maybeWrap(err)
	}

	return rsp, nil
}

func (c *client) GetLoadStatisticsEntry(ctx context.Context, time time.Time) (*awacspb.GetLoadStatisticsEntryResponse, error) {
	req := &awacspb.GetLoadStatisticsEntryRequest{
		Start: timestamppb.New(time),
	}

	rsp, err := c.c.GetLoadStatisticsEntry(ctx, req)
	if err != nil {
		return nil, c.maybeWrap(err)
	}
	return rsp, nil
}

func (c *client) UpdateBackend(ctx context.Context, specPb *awacspb.BackendSpec, namespaceID, backendID, version, comment string) (*awacspb.UpdateBackendResponse, error) {
	req := awacspb.UpdateBackendRequest{
		Meta: &awacspb.BackendMeta{
			Id:          backendID,
			NamespaceId: namespaceID,
			Comment:     comment,
			Version:     version,
		},
		Spec: specPb,
	}

	rsp, err := c.c.UpdateBackend(ctx, &req)
	if err != nil {
		return nil, c.maybeWrap(err)
	}

	return rsp, nil
}

func (c *client) GetNamespace(ctx context.Context, namespaceID string) (*awacspb.Namespace, error) {
	req := &awacspb.GetNamespaceRequest{Id: namespaceID}
	rsp, err := c.c.GetNamespace(ctx, req)
	if err != nil {
		return nil, c.maybeWrap(err)
	}
	return rsp.GetNamespace(), nil
}
