package awacs

import (
	awacspb "a.yandex-team.ru/infra/awacs/proto"
	"time"
)

func findBalancerRevisionStatus(statuses []*awacspb.BalancerRevisionStatus, cond func(*awacspb.BalancerRevisionStatus) bool) *awacspb.BalancerRevisionStatus {
	for _, status := range statuses {
		if cond(status) {
			return status
		}
	}
	return nil
}

func findBalancerRevisionStatusMany(statuses []*awacspb.BalancerRevisionStatus, cond func(*awacspb.BalancerRevisionStatus) bool) []*awacspb.BalancerRevisionStatus {
	var rv []*awacspb.BalancerRevisionStatus
	for _, status := range statuses {
		if cond(status) {
			rv = append(rv, status)
		}
	}
	return rv
}

func findActiveBalancerRevisionStatus(statuses []*awacspb.BalancerRevisionStatus) *awacspb.BalancerRevisionStatus {
	return findBalancerRevisionStatus(statuses, func(status *awacspb.BalancerRevisionStatus) bool {
		active := status.GetActive()
		return active != nil && active.Status == "True"
	})
}

func findInProgressBalancerRevisionStatusMany(statuses []*awacspb.BalancerRevisionStatus) []*awacspb.BalancerRevisionStatus {
	return findBalancerRevisionStatusMany(statuses, func(status *awacspb.BalancerRevisionStatus) bool {
		inProgress := status.GetInProgress()
		return inProgress != nil && inProgress.Status == "True"
	})
}

const (
	Balancer    = iota
	Domain      = iota
	Certificate = iota
	Upstream    = iota
	Backend     = iota
	EndpointSet = iota
	Knob        = iota
)

var BalancerStateEntryTypeNames = []string{"balancer", "domain", "certificate", "upstream", "backend", "endpointset", "knob"}

type BalancerStateEntry struct {
	Type     int
	FlatID   string
	Statuses []*awacspb.BalancerState_RevisionStatus
}

func listBalancerStateEntries(state *awacspb.BalancerState) []BalancerStateEntry {
	var rv []BalancerStateEntry
	rv = append(rv, BalancerStateEntry{
		Balancer,
		state.NamespaceId + "/" + state.BalancerId,
		state.GetBalancer().GetStatuses(),
	})
	for domainID, statuses := range state.GetDomains() {
		rv = append(rv, BalancerStateEntry{
			Domain,
			state.NamespaceId + "/" + domainID,
			statuses.GetStatuses(),
		})
	}
	for certificateID, statuses := range state.GetCertificates() {
		rv = append(rv, BalancerStateEntry{
			Certificate,
			state.NamespaceId + "/" + certificateID,
			statuses.GetStatuses(),
		})
	}
	for upstreamID, statuses := range state.GetUpstreams() {
		rv = append(rv, BalancerStateEntry{
			Upstream,
			state.NamespaceId + "/" + upstreamID,
			statuses.GetStatuses(),
		})
	}
	for backendID, statuses := range state.GetBackends() {
		rv = append(rv, BalancerStateEntry{
			Backend,
			state.NamespaceId + "/" + backendID,
			statuses.GetStatuses(),
		})
	}
	for endpointSetID, statuses := range state.GetEndpointSets() {
		rv = append(rv, BalancerStateEntry{
			EndpointSet,
			state.NamespaceId + "/" + endpointSetID,
			statuses.GetStatuses(),
		})
	}
	for knobID, statuses := range state.GetKnobs() {
		rv = append(rv, BalancerStateEntry{
			Knob,
			state.NamespaceId + "/" + knobID,
			statuses.GetStatuses(),
		})
	}
	return rv
}

func findCertificateRevisionStatusPerBalancer(statuses []*awacspb.CertificateRevisionStatusPerBalancer, cond func(*awacspb.CertificateRevisionStatusPerBalancer) bool) *awacspb.CertificateRevisionStatusPerBalancer {
	for _, status := range statuses {
		if cond(status) {
			return status
		}
	}
	return nil
}

func findCertificateRevisionStatusPerBalancerByID(statuses []*awacspb.CertificateRevisionStatusPerBalancer, id string) *awacspb.CertificateRevisionStatusPerBalancer {
	return findCertificateRevisionStatusPerBalancer(statuses, func(status *awacspb.CertificateRevisionStatusPerBalancer) bool {
		return status.Id == id
	})
}

func isBalancerStuck(balancer *awacspb.Balancer, hasBeenStuck bool, state *awacspb.BalancerState) bool {
	const stuckInProgressTime = time.Hour
	const settleTime = time.Minute * 15
	if balancer.Spec.Incomplete {
		return false
	}

	if !hasBeenStuck {
		// We do not want to consider balancer as stuck only if snapshot ctime is too old - it can be zerodiff
		// using the same snapshot as previous revision. So let's check revision ctime. But if balancer is already
		// considered as stuck - skip this check to avoid flaps.
		firstNonActiveRevisionCtime := time.Now().Add(-1 * settleTime)
		for _, entry := range listBalancerStateEntries(state) {
			for _, status := range entry.Statuses {
				active := status.GetActive().Status
				if active == "True" {
					continue
				}
				if firstNonActiveRevisionCtime.Before(status.Ctime.AsTime()) {
					firstNonActiveRevisionCtime = status.Ctime.AsTime()
				}
			}
		}
		if time.Since(firstNonActiveRevisionCtime) < settleTime {
			return false
		}
	}

	for _, revision := range balancer.Status.Revisions {
		if revision.InProgress.Status == "False" {
			continue
		}
		for _, snapshot := range revision.InProgress.Meta.GetNannyStaticFile().Snapshots {
			if time.Since(snapshot.Ctime.AsTime()) > stuckInProgressTime {
				return true
			}
		}
	}
	return false
}

func isL3BalancerStuck(l3Balancer *awacspb.L3Balancer) bool {
	const stuckInProgressTime = time.Hour
	const settleTime = time.Minute * 10
	if l3Balancer.Spec.Incomplete {
		return false
	}
	for _, revision := range l3Balancer.L3Status.Revisions {
		if revision.InProgress.Status == "False" {
			continue
		}
		if time.Since(revision.Ctime.AsTime()) < settleTime {
			continue
		}
		for _, config := range revision.InProgress.Meta.GetL3Mgr().Configs {
			if time.Since(config.Ctime.AsTime()) > stuckInProgressTime {
				return true
			}
		}
	}
	return false
}
