package calendar

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"strings"
	"time"
)

const (
	HolidayTypeHoliday = "holiday"
	HolidayTypeWeekend = "weekend"
)

type HolidayDate time.Time

func (date *HolidayDate) AsTime() time.Time {
	return time.Time(*date)
}
func (date *HolidayDate) UnmarshalJSON(b []byte) error {
	s := strings.Trim(string(b), "\"")
	loc, _ := time.LoadLocation("Europe/Moscow")
	t, err := time.ParseInLocation("2006-01-02", s, loc)
	if err != nil {
		return err
	}
	*date = HolidayDate(t)
	return nil
}

func DateStr(t time.Time) string {
	y, m, d := t.Date()
	return fmt.Sprintf("%d-%d-%d", y, m, d)
}

type Holiday struct {
	Date HolidayDate `json:"date"`
	Type string      `json:"type"`
	Name string      `json:"name"`
}

type HolidaysResp struct {
	Holidays []Holiday
}

func ListHolidays(startDate, endDate time.Time) ([]HolidayDate, error) {
	client := &http.Client{}
	client.Timeout = time.Second * 2

	startDateStr := DateStr(startDate)
	endDateStr := DateStr(endDate)

	query := fmt.Sprintf("https://api.calendar.yandex-team.ru/intapi/get-holidays?from=%s&to=%s&for=rus", startDateStr, endDateStr)
	req, err := http.NewRequest("GET", query, nil)
	if err != nil {
		return nil, err
	}

	resp, err := client.Do(req)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != 200 {
		return nil, fmt.Errorf("bad HTTP status code %d", resp.StatusCode)
	}

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, err
	}

	var holidays HolidaysResp
	err = json.Unmarshal(body, &holidays)
	if err != nil {
		return nil, err
	}

	result := make([]HolidayDate, 0)

	for _, h := range holidays.Holidays {
		if h.Type != HolidayTypeHoliday && h.Type != HolidayTypeWeekend {
			continue
		}
		result = append(result, h.Date)
	}

	return result, nil
}
