package pods

import (
	"context"
	"errors"
	"fmt"
	"time"

	"go.temporal.io/sdk/log"

	"a.yandex-team.ru/infra/nanny2/pkg/ypclient"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"go.temporal.io/sdk/activity"
)

var ActiveState = "ACTIVE"

type PodInfo struct {
	PodID    string
	Cluster  string
	NodeID   string
	Itype    string
	HostName string
}

type Eviction struct {
	State       ypapi.EEvictionState
	Reason      ypapi.EEvictionReason
	LastUpdated time.Time
}

type Maintenance struct {
	State ypapi.EPodMaintenanceState
}

type Pod struct {
	PodInfo     *PodInfo
	State       string
	Eviction    *Eviction
	Maintenance *Maintenance
}

type Activities struct {
	ypClient ypclient.YpClientInterface
	clusters []string
}

func NewActivities(ypToken string, clusters []string) (*Activities, error) {
	ypClient, err := ypclient.NewYpClientToken(clusters, 200, ypToken)
	if err != nil {
		return nil, err
	}

	return &Activities{
		ypClient: ypClient,
	}, nil
}

func getPodTargetActiveConf(podStatus *ypapi.TPodStatus) *string {
	for instance, summary := range podStatus.GetIssConfSummaries() {
		if *summary.TargetState == ActiveState {
			return &instance
		}
	}

	return nil
}

func getPodInfo(ypPod *ypclient.YpPod) (*Pod, error) {
	targetActiveConf := getPodTargetActiveConf(ypPod.Pod.GetStatus())
	if targetActiveConf == nil {
		return nil, errors.New("don't track pods without targetState = ACTIVE")
	}

	summaries := ypPod.Pod.GetStatus().GetAgent().GetIssSummary().GetStateSummaries()
	if summaries == nil {
		return nil, errors.New("summaries field is empty")
	}
	instanceSummary, ok := summaries[*targetActiveConf]
	if !ok || instanceSummary == nil {
		return nil, errors.New("instnace summary value is empty or doesn't exist")
	}

	instances := ypPod.Pod.GetSpec().GetIss().GetInstances()
	itype := "unknown"
	if len(instances) != 0 {
		itype = instances[0].Properties["INSTANCE_TAG_ITYPE"]
	}

	pod := Pod{
		PodInfo: &PodInfo{
			PodID:   ypPod.Pod.Meta.Id,
			Cluster: ypPod.Cluster,
			NodeID:  *ypPod.Pod.Spec.NodeId,
			Itype:   itype,
		},
		State: *instanceSummary.CurrentState,
		Eviction: &Eviction{
			State:       ypPod.Pod.GetStatus().GetEviction().GetState(),
			Reason:      ypPod.Pod.GetStatus().GetEviction().GetReason(),
			LastUpdated: time.Unix(0, int64(ypPod.Pod.GetStatus().GetEviction().GetLastUpdated())*1000),
		},
		Maintenance: &Maintenance{
			State: ypPod.Pod.GetStatus().GetMaintenance().GetState(),
		},
	}
	pod.PodInfo.HostName = fmt.Sprintf("%s.%s.yp-c.yandex.net", pod.PodInfo.PodID, pod.PodInfo.Cluster)

	return &pod, nil
}

func IsEvictionRequested(eviction *Eviction, maintenance *Maintenance) bool {
	if eviction.State != ypapi.EEvictionState_ES_REQUESTED {
		return false
	}

	if eviction.Reason == ypapi.EEvictionReason_ER_HFSM && maintenance.State != ypapi.EPodMaintenanceState_PMS_REQUESTED {
		// in that case eviction is not really requested
		return false
	}

	return true
}

func includePod(pod *Pod, faultySet map[string]interface{}) bool {
	if _, ok := faultySet[pod.PodInfo.HostName]; ok || pod.State != ActiveState {
		return true
	}

	return IsEvictionRequested(pod.Eviction, pod.Maintenance)
}

func getPods(ypPods []*ypclient.YpPod, logger log.Logger) []*Pod {
	pods := make([]*Pod, 0, len(ypPods))
	for _, ypPod := range ypPods {
		pod, err := getPodInfo(ypPod)
		if err != nil {
			logger.Info(fmt.Sprintf("skipping pod %s: %v", ypPod.Pod.GetMeta().GetId(), err))
			continue
		}
		pods = append(pods, pod)
	}
	return pods
}

func filterPods(pods []*Pod, faultyPodNames []string) map[string]*Pod {
	faultySet := make(map[string]interface{})
	for _, podName := range faultyPodNames {
		faultySet[podName] = nil
	}

	podMap := make(map[string]*Pod)
	for _, pod := range pods {
		if includePod(pod, faultySet) {
			podMap[pod.PodInfo.HostName] = pod
		}
	}

	return podMap
}

type PodsForServiceResponse struct {
	Pods           map[string]*Pod
	TotalPodsCount int
}

// returns mapping HostName : *Pod
func (a *Activities) GetPodsForServiceActivity(ctx context.Context, serviceID string, faultyPodNames []string) (*PodsForServiceResponse, error) {
	logger := activity.GetLogger(ctx)

	ypPods, err := a.ypClient.ListServicePods(ctx, serviceID)
	if err != nil {
		return nil, err
	}

	pods := getPods(ypPods, logger)
	return &PodsForServiceResponse{
		Pods:           filterPods(pods, faultyPodNames),
		TotalPodsCount: len(ypPods),
	}, nil
}
