package pods

import (
	"context"
	"testing"
	"time"

	"a.yandex-team.ru/infra/nanny2/pkg/ypclient"
	"a.yandex-team.ru/yp/go/proto/clusterapi"
	"a.yandex-team.ru/yp/go/proto/ypapi"
	"github.com/stretchr/testify/suite"
	"go.temporal.io/sdk/testsuite"
)

type PodsTestSuite struct {
	suite.Suite
	testsuite.WorkflowTestSuite

	env *testsuite.TestActivityEnvironment
}

func (suite *PodsTestSuite) SetupTest() {
	suite.env = suite.NewTestActivityEnvironment()
}

func (suite *PodsTestSuite) getPodsForServiceExec(activity interface{}, faultyPodNames []string) *PodsForServiceResponse {
	var rsp *PodsForServiceResponse
	eval, err := suite.env.ExecuteActivity(activity, "test", []string{})
	suite.Require().NoError(err)
	err = eval.Get(&rsp)
	suite.Require().NoError(err)

	return rsp
}

func (suite *PodsTestSuite) TestGetPodTargetActiveConf() {
	notActiveTargetState := "OTHER"
	podStatus := ypapi.TPodStatus{
		IssConfSummaries: map[string]*ypapi.TPodStatus_TIssConfSummary{
			"instance1": {TargetState: &notActiveTargetState},
			"instance2": {TargetState: &ActiveState},
		},
	}

	resp := getPodTargetActiveConf(&podStatus)
	suite.Require().NotNil(resp)
	suite.Equal("instance2", *resp)
}

func (suite *PodsTestSuite) TestGetPodInfo() {
	nodeID := "nodeID"
	var evictionLastUpdated uint64 = 1
	ypPod := ypclient.YpPod{
		Cluster: "sas",
		Pod: ypapi.TPod{
			Status: &ypapi.TPodStatus{
				IssConfSummaries: map[string]*ypapi.TPodStatus_TIssConfSummary{
					"instance": {TargetState: &ActiveState},
				},
				Agent: &ypapi.TPodStatus_TAgent{
					IssSummary: &ypapi.TPodStatus_TAgent_TIssSummary{
						StateSummaries: map[string]*ypapi.TPodStatus_TAgent_TIssStateSummary{
							"instance": {CurrentState: &ActiveState},
						},
					},
				},
				Eviction: &ypapi.TPodStatus_TEviction{
					State:       ypapi.EEvictionState_ES_REQUESTED.Enum(),
					Reason:      ypapi.EEvictionReason_ER_HFSM.Enum(),
					LastUpdated: &evictionLastUpdated,
				},
				Maintenance: &ypapi.TPodStatus_TMaintenance{
					State: ypapi.EPodMaintenanceState_PMS_IN_PROGRESS.Enum(),
				},
			},
			Spec: &ypapi.TPodSpec{
				Iss: &clusterapi.HostConfiguration{
					Instances: []*clusterapi.HostConfigurationInstance{
						{Properties: map[string]string{"INSTANCE_TAG_ITYPE": "itype"}},
					},
				},
				NodeId: &nodeID,
			},
			Meta: &ypapi.TPodMeta{Id: "podID"},
		},
	}

	pod, err := getPodInfo(&ypPod)
	suite.Require().NoError(err)

	suite.Equal(
		&Pod{
			PodInfo: &PodInfo{
				PodID:    "podID",
				Cluster:  "sas",
				NodeID:   "nodeID",
				Itype:    "itype",
				HostName: "podID.sas.yp-c.yandex.net",
			},
			State: "ACTIVE",
			Eviction: &Eviction{
				State:       ypapi.EEvictionState_ES_REQUESTED,
				Reason:      ypapi.EEvictionReason_ER_HFSM,
				LastUpdated: time.Unix(0, 1000),
			},
			Maintenance: &Maintenance{
				State: ypapi.EPodMaintenanceState_PMS_IN_PROGRESS,
			},
		},
		pod,
	)

}

func (suite *PodsTestSuite) TestFilterPods() {
	faultyPodNames := []string{"pod1"}
	pods := []*Pod{
		// true: in faultyPodNames
		{PodInfo: &PodInfo{HostName: "pod1"}},
		// true: this pod is faulty
		{PodInfo: &PodInfo{HostName: "pod2"}, State: "BAD"},
		// false: pos is ok
		{PodInfo: &PodInfo{HostName: "pod3"}, State: ActiveState, Eviction: &Eviction{}},
		// true: eviction requested
		{
			PodInfo:     &PodInfo{HostName: "pod4"},
			State:       ActiveState,
			Eviction:    &Eviction{State: ypapi.EEvictionState_ES_REQUESTED},
			Maintenance: &Maintenance{},
		},
		// false: hfsm and maintenance not requested
		{
			PodInfo: &PodInfo{HostName: "pod5"},
			State:   ActiveState,
			Eviction: &Eviction{
				State:  ypapi.EEvictionState_ES_REQUESTED,
				Reason: ypapi.EEvictionReason_ER_HFSM,
			},
			Maintenance: &Maintenance{State: ypapi.EPodMaintenanceState_PMS_NONE},
		},
	}

	filtered := filterPods(pods, faultyPodNames)
	suite.Equal(
		map[string]*Pod{
			"pod1": pods[0],
			"pod2": pods[1],
			"pod4": pods[3],
		},
		filtered,
	)
}

func (suite *PodsTestSuite) TestGetPodsForServiceEmptyPods() {
	a := Activities{
		ypClient: &ypclient.FakeYpClient{
			OnListServicePods: func(ctx context.Context, serviceID string) ([]*ypclient.YpPod, error) {
				return []*ypclient.YpPod{}, nil
			},
		},
	}
	suite.env.RegisterActivity(a.GetPodsForServiceActivity)

	rsp := suite.getPodsForServiceExec(a.GetPodsForServiceActivity, []string{})
	suite.Equal(0, rsp.TotalPodsCount)
	suite.Equal(0, len(rsp.Pods))
}

func (suite *PodsTestSuite) TestGetPodsForServiceActivity() {
	nodeID := "nodeID"
	currentState := "notactive"
	var evictionLastUpdated uint64 = 1
	ypPod := ypclient.YpPod{
		Cluster: "sas",
		Pod: ypapi.TPod{
			Status: &ypapi.TPodStatus{
				IssConfSummaries: map[string]*ypapi.TPodStatus_TIssConfSummary{
					"instance": {TargetState: &ActiveState},
				},
				Agent: &ypapi.TPodStatus_TAgent{
					IssSummary: &ypapi.TPodStatus_TAgent_TIssSummary{
						StateSummaries: map[string]*ypapi.TPodStatus_TAgent_TIssStateSummary{
							"instance": {CurrentState: &currentState},
						},
					},
				},
				Eviction: &ypapi.TPodStatus_TEviction{
					State:       ypapi.EEvictionState_ES_REQUESTED.Enum(),
					Reason:      ypapi.EEvictionReason_ER_HFSM.Enum(),
					LastUpdated: &evictionLastUpdated,
				},
				Maintenance: &ypapi.TPodStatus_TMaintenance{
					State: ypapi.EPodMaintenanceState_PMS_IN_PROGRESS.Enum(),
				},
			},
			Spec: &ypapi.TPodSpec{
				Iss: &clusterapi.HostConfiguration{
					Instances: []*clusterapi.HostConfigurationInstance{
						{Properties: map[string]string{"INSTANCE_TAG_ITYPE": "itype"}},
					},
				},
				NodeId: &nodeID,
			},
			Meta: &ypapi.TPodMeta{Id: "podID"},
		},
	}

	a := Activities{
		ypClient: &ypclient.FakeYpClient{
			OnListServicePods: func(ctx context.Context, serviceID string) ([]*ypclient.YpPod, error) {
				return []*ypclient.YpPod{&ypPod, {}, {}}, nil
			},
		},
	}
	suite.env.RegisterActivity(a.GetPodsForServiceActivity)

	rsp := suite.getPodsForServiceExec(a.GetPodsForServiceActivity, []string{})
	suite.Equal(1, len(rsp.Pods))
	suite.Equal(3, rsp.TotalPodsCount)
}

func TestPodsTestSuite(t *testing.T) {
	suite.Run(t, new(PodsTestSuite))
}
