package services

import (
	"context"
	"regexp"
	"strings"

	pb "a.yandex-team.ru/infra/nanny/go/proto/nanny_repo"
	nannyClient "a.yandex-team.ru/infra/temporal/clients/nanny"
)

const ActiveState = "ACTIVE"

type Diff struct {
	Created []string
	Deleted []string
}

type Activities struct {
	nannyClient *nannyClient.Client
}

func NewActivities(nannyURL string, oauthToken string, rpcClient pb.RepoServiceClient) *Activities {
	return &Activities{
		nannyClient: nannyClient.NewClient(nannyURL, oauthToken, rpcClient),
	}
}

func filterByRegexps(services []*nannyClient.Service, regexps []string) ([]*nannyClient.Service, error) {
	if len(regexps) == 0 {
		return services, nil
	}

	var filteredServices []*nannyClient.Service
	for _, service := range services {
		for _, r := range regexps {
			matched, err := regexp.MatchString(r, service.ID)
			if err != nil {
				return nil, err
			}
			if matched {
				filteredServices = append(filteredServices, service)
				break
			}
		}
	}

	return filteredServices, nil
}

func filterByDeployEngines(services []*nannyClient.Service, engine string) []*nannyClient.Service {
	if engine == "" {
		return services
	}

	var filteredServices []*nannyClient.Service
	for _, service := range services {
		if service.RuntimeAttrs.MetaInfo.Annotaions.DeployEngine == engine {
			filteredServices = append(filteredServices, service)
		}
	}

	return filteredServices
}

func (a *Activities) GetServiceIdsActivity(ctx context.Context, skip int, limit int) ([]string, error) {
	serviceSummaries, err := a.nannyClient.ListSummaries(
		ctx,
		nannyClient.ListSummariesRequest{
			IncludeLabels: false,
			Limit:         limit,
			Skip:          skip,
		},
	)
	if err != nil {
		return nil, err
	}

	var services []string
	for _, serviceSummary := range serviceSummaries {
		services = append(services, serviceSummary.ServiceId)
	}

	return services, nil
}

type GetServicesRequest struct {
	Skip         int
	Limit        int
	Regexps      []string
	DeployEngine string
}

func (a *Activities) GetServicesActivity(ctx context.Context, r *GetServicesRequest) ([]*nannyClient.Service, error) {
	services, err := a.nannyClient.GetServices(ctx, r.Skip, r.Limit)
	if err != nil {
		return nil, err
	}

	services, err = filterByRegexps(services, r.Regexps)
	if err != nil {
		return nil, err
	}

	return filterByDeployEngines(services, r.DeployEngine), nil
}

func (a *Activities) GetServiceInfoAttrsActivity(
	ctx context.Context,
	serviceID string) (*nannyClient.ServiceInfoAttrs, error) {
	return a.nannyClient.GetServiceInfoAttrs(serviceID)
}

func (a *Activities) GetServiceLatestRuntimeAttrsAuthors(ctx context.Context, serviceID string) ([]string, error) {
	runtimeAttrsHistory, err := a.nannyClient.GetServiceRuntimeAttrsHistory(serviceID, 10)
	if err != nil {
		return nil, err
	}

	var uniqueLogins []string
	loginsMap := make(map[string]interface{})
	for _, info := range runtimeAttrsHistory {
		login := info.ChangeInfo.Author
		if _, ok := loginsMap[login]; !ok && !strings.Contains(login, "robot") {
			uniqueLogins = append(uniqueLogins, login)
			loginsMap[login] = nil
		}
	}

	return uniqueLogins, nil
}

func (a *Activities) GetService(ctx context.Context, serviceID string) (*pb.Service, error) {
	return a.nannyClient.GetService(ctx, serviceID, []string{"status.snapshot", "status.summary.value"})
}

func (a *Activities) GetServiceReplicationPolicy(ctx context.Context, serviceID string) (*pb.ReplicationPolicy, error) {
	return a.nannyClient.GetServiceReplicationPolicy(ctx, serviceID)
}
