package abc

import (
	"fmt"
	"net/http"
	"strconv"
	"time"

	"github.com/go-resty/resty/v2"
)

const (
	defaultURL            = "https://abc-back.yandex-team.ru/api/v4/"
	defaultRequestTimeout = time.Second * 30
)

type ClientConfig struct {
	URL            string        `yaml:"url"`
	OauthToken     string        `yaml:"token"`
	RequestTimeout time.Duration `yaml:"request_timeout"`
}

type Client struct {
	c *resty.Client
}

func NewClient(config *ClientConfig) *Client {
	requestTimeout := config.RequestTimeout
	if requestTimeout == 0 {
		requestTimeout = defaultRequestTimeout
	}

	URL := config.URL
	if URL == "" {
		URL = defaultURL
	}

	return &Client{
		c: resty.New().
			SetTimeout(requestTimeout).
			SetHostURL(URL).
			SetAuthScheme("OAuth").
			SetAuthToken(config.OauthToken).
			SetHeader("Content-Type", "application/json").
			SetHeader("Accept", "application/json"),
	}
}

type Person struct {
	Login string `json:"login,omitempty"`
}

type Duty struct {
	Person Person `json:"person,omitempty"`
}

func uniquifyLogins(logins []string) []string {
	var uniqueLogins []string
	loginsMap := make(map[string]interface{})
	for _, login := range logins {
		if _, ok := loginsMap[login]; !ok && login != "" {
			uniqueLogins = append(uniqueLogins, login)
			loginsMap[login] = nil
		}
	}

	return uniqueLogins
}

func (c *Client) onDuty(params map[string]string) ([]string, error) {
	dutyPersons := make([]Duty, 0)
	resp, err := c.c.R().SetQueryParams(params).SetResult(&dutyPersons).Get("/duty/on_duty/")
	if err != nil {
		return nil, err
	}

	if resp.StatusCode() != http.StatusOK {
		return nil, fmt.Errorf("unsupported response code from service: %d: %s", resp.StatusCode(), resp.String())
	}

	rv := make([]string, 0)
	for _, duty := range dutyPersons {
		rv = append(rv, duty.Person.Login)
	}
	return uniquifyLogins(rv), nil
}

func (c *Client) GetOnDutyFromSchedule(abcScheduleID int) ([]string, error) {
	params := map[string]string{"schedule": strconv.Itoa(abcScheduleID), "fields": "person.login"}
	rv, err := c.onDuty(params)
	return rv, err
}

func (c *Client) GetOnDutyFromService(abcServiceID int) ([]string, error) {
	params := map[string]string{"service": strconv.Itoa(abcServiceID), "fields": "person.login"}
	rv, err := c.onDuty(params)
	return rv, err
}

type ServiceMembersResult struct {
	Person Person `json:"person"`
}

type GetServiceMembersResponse struct {
	Results []ServiceMembersResult `json:"results"`
}

func (c *Client) getServiceMembersLogins(params map[string]string) ([]string, error) {
	params["fields"] = "person.login"

	var members GetServiceMembersResponse
	resp, err := c.c.R().SetQueryParams(params).SetResult(&members).Get("/services/members/")
	if err != nil {
		return nil, err
	}

	if resp.StatusCode() != http.StatusOK {
		return nil, fmt.Errorf("unsupported response code from service: %d: %s", resp.StatusCode(), resp.String())
	}

	logins := make([]string, 0)
	for _, result := range members.Results {
		logins = append(logins, result.Person.Login)
	}

	return uniquifyLogins(logins), nil
}

func (c *Client) GetServiceMembers(abcServiceID int) ([]string, error) {
	params := map[string]string{
		"service":  strconv.Itoa(abcServiceID),
		"is_robot": "false",
	}
	return c.getServiceMembersLogins(params)
}

func (c *Client) GetServiceMembersBySlug(abcServiceSlug string) ([]string, error) {
	params := map[string]string{
		"service__slug": abcServiceSlug,
		"is_robot":      "false",
	}
	return c.getServiceMembersLogins(params)
}
