package client

import (
	"context"
	"fmt"
	"net/http"
	"strconv"

	"github.com/go-resty/resty/v2"
	"google.golang.org/protobuf/types/known/fieldmaskpb"

	pb "a.yandex-team.ru/infra/nanny/go/proto/nanny_repo"
)

type ListSummariesRequest struct {
	IncludeLabels bool
	Skip          int
	Limit         int
	IDRegexp      string
}

type GetServiceInfoAttrsResponse struct {
	Content ServiceInfoAttrs `json:"content"`
}

type GetServiceRuntimeAttrsHistoryResponse struct {
	Result []*ServiceRuntimeAttrsInfo `json:"result"`
}

type GetServicesResponse struct {
	Result []*Service `json:"result"`
}

type Config struct {
	URL        string
	OAuthToken string
}

type Client struct {
	restClient *resty.Client
	rpcClient  pb.RepoServiceClient
}

func NewClient(URL string, oauthToken string, repoClient pb.RepoServiceClient) *Client {
	return &Client{
		restClient: resty.New().
			SetHostURL(URL).
			SetAuthScheme("OAuth").
			SetAuthToken(oauthToken).
			SetHeader("Content-Type", "application/json").
			SetHeader("Accept", "application/json"),
		rpcClient: repoClient,
	}
}

func (c *Client) ListSummaries(ctx context.Context, req ListSummariesRequest) ([]*pb.ServiceSummary, error) {
	var request = pb.ListSummariesRequest{
		IncludeLabels: req.IncludeLabels,
		Skip:          int32(req.Skip),
		Limit:         int32(req.Limit),
		IdRegexp:      req.IDRegexp,
	}
	response, err := c.rpcClient.ListSummaries(ctx, &request)
	if err != nil {
		return nil, err
	}

	return response.Value, nil
}

func (c *Client) GetService(ctx context.Context, serviceID string, fieldMask []string) (*pb.Service, error) {
	var request = pb.GetServiceRequest{
		ServiceId: serviceID,
		FieldMask: &fieldmaskpb.FieldMask{
			Paths: fieldMask,
		},
	}
	response, err := c.rpcClient.GetService(ctx, &request)
	if err != nil {
		return nil, err
	}

	return response.Service, nil
}

func (c *Client) GetServices(ctx context.Context, skip int, limit int) ([]*Service, error) {
	const BatchSize = 200

	var services []*Service
	for i := 0; i*BatchSize < limit; i += 1 {
		resp, err := c.restClient.R().
			SetResult(&GetServicesResponse{}).
			SetQueryParams(
				map[string]string{
					"skip":  strconv.Itoa(skip + i*BatchSize),
					"limit": strconv.Itoa(BatchSize),
				},
			).
			Get("v2/services/")

		if err != nil {
			return nil, err
		}

		if resp.StatusCode() != http.StatusOK {
			return nil, fmt.Errorf("unsupported response code from service: %d: %s", resp.StatusCode(), resp.String())
		}

		services = append(services, resp.Result().(*GetServicesResponse).Result...)
	}

	if len(services) > limit {
		services = services[:limit]
	}

	return services, nil
}

func (c *Client) GetServiceInfoAttrs(serviceID string) (*ServiceInfoAttrs, error) {
	resp, err := c.restClient.R().
		SetResult(&GetServiceInfoAttrsResponse{}).
		Get(fmt.Sprintf("v2/services/%s/info_attrs/", serviceID))

	if err != nil {
		return nil, err
	}

	if resp.StatusCode() == http.StatusNotFound {
		return nil, &ServiceNotFoundError{"service not found"}
	}

	if resp.StatusCode() != http.StatusOK {
		return nil, fmt.Errorf("unsupported response code from service: %d: %s", resp.StatusCode(), resp.String())
	}

	return &resp.Result().(*GetServiceInfoAttrsResponse).Content, nil
}

func (c *Client) GetServiceRuntimeAttrsHistory(serviceID string, limit int) ([]*ServiceRuntimeAttrsInfo, error) {
	resp, err := c.restClient.R().
		SetResult(&GetServiceRuntimeAttrsHistoryResponse{}).
		Get(fmt.Sprintf("v2/services/%s/history/runtime_attrs_infos/?limit=%d&skip=0", serviceID, limit))

	if err != nil {
		return nil, err
	}

	if resp.StatusCode() != http.StatusOK {
		return nil, fmt.Errorf("unsupported response code from service: %d: %s", resp.StatusCode(), resp.String())
	}

	return resp.Result().(*GetServiceRuntimeAttrsHistoryResponse).Result, nil
}

func (c *Client) GetServiceReplicationPolicy(ctx context.Context, policyID string) (*pb.ReplicationPolicy, error) {
	request := &pb.GetReplicationPolicyRequest{PolicyId: policyID}
	resp, err := c.rpcClient.GetReplicationPolicy(ctx, request)
	if err != nil {
		return nil, err
	}

	return resp.GetPolicy(), nil
}
