package staff

import (
	"fmt"
	"net/http"
	"strings"
	"time"

	"github.com/go-resty/resty/v2"
)

const (
	defaultURL            = "https://staff-api.yandex-team.ru/v3/"
	defaultRequestTimeout = time.Second * 30
)

type ClientConfig struct {
	URL            string        `yaml:"url"`
	OauthToken     string        `yaml:"token"`
	RequestTimeout time.Duration `yaml:"request_timeout"`
}

type Client struct {
	c *resty.Client
}

func NewClient(config *ClientConfig) *Client {
	requestTimeout := config.RequestTimeout
	if requestTimeout == 0 {
		requestTimeout = defaultRequestTimeout
	}

	URL := config.URL
	if URL == "" {
		URL = defaultURL
	}

	return &Client{
		c: resty.New().
			SetTimeout(requestTimeout).
			SetHostURL(URL).
			SetAuthScheme("OAuth").
			SetAuthToken(config.OauthToken).
			SetHeader("Content-Type", "application/json").
			SetHeader("Accept", "application/json"),
	}
}

type ListPersonsRequest struct {
	Login string `json:"login,omitempty"`
}

type Official struct {
	IsDismissed bool `json:"is_dismissed,omitempty"`
	IsRobot     bool `json:"is_robot,omitempty"`
}

type Person struct {
	ID        int       `json:"id,omitempty"`
	Login     string    `json:"login,omitempty"`
	IsDeleted bool      `json:"is_deleted,omitempty"`
	Official  *Official `json:"official,omitempty"`
}

type ListPersonsResponse struct {
	Result []*Person `json:"result,omitempty"`
}

type GroupPersonResponse struct {
	Person Person `json:"person,omitempty"`
}

type ListGroupsMembersResponse struct {
	Result []*GroupPersonResponse `json:"result,omitempty"`
}

func (c *Client) ListPersons(logins []string) ([]*Person, error) {
	params := map[string]string{"login": strings.Join(logins, ",")}
	resp, err := c.c.R().SetQueryParams(params).SetResult(&ListPersonsResponse{}).Get("/persons/")
	if err != nil {
		return nil, err
	}

	if resp.StatusCode() != http.StatusOK {
		return nil, fmt.Errorf("unsupported response code from service: %d: %s", resp.StatusCode(), resp.String())
	}

	return resp.Result().(*ListPersonsResponse).Result, nil
}

func (c *Client) ListGroupsMembers(groupIDs []string) ([]*Person, error) {
	params := map[string]string{"group.id": strings.Join(groupIDs, ",")}
	resp, err := c.c.R().SetQueryParams(params).SetResult(&ListGroupsMembersResponse{}).Get("/groupmembership/")
	if err != nil {
		return nil, err
	}

	if resp.StatusCode() != http.StatusOK {
		return nil, fmt.Errorf("unsupported response code from service: %d: %s", resp.StatusCode(), resp.String())
	}

	result := resp.Result().(*ListGroupsMembersResponse).Result
	rv := make([]*Person, 0)
	for _, person := range result {
		rv = append(rv, &person.Person)
	}
	return rv, nil
}

func FilterActualPersonsLogins(persons []*Person) []string {
	rv := make([]string, 0)
	for _, person := range persons {
		if person.Official.IsDismissed || person.Official.IsRobot || person.IsDeleted {
			continue
		}
		rv = append(rv, person.Login)
	}
	return rv
}
