package startrek

import (
	"fmt"
	"net/http"
	"time"

	"github.com/go-resty/resty/v2"
)

type ClientConfig struct {
	URL            string        `yaml:"url"`
	OauthToken     string        `yaml:"token"`
	RequestTimeout time.Duration `yaml:"request_timeout"`
}

type Client struct {
	c *resty.Client
}

const (
	defaultURL            = "https://st-api.yandex-team.ru/v2/"
	defaultRequestTimeout = time.Second * 30
)

func NewClient(config *ClientConfig) *Client {
	requestTimeout := config.RequestTimeout
	if requestTimeout == 0 {
		requestTimeout = defaultRequestTimeout
	}

	URL := config.URL
	if URL == "" {
		URL = defaultURL
	}

	return &Client{
		c: resty.New().
			SetTimeout(requestTimeout).
			SetBaseURL(URL).
			SetAuthScheme("OAuth").
			SetAuthToken(config.OauthToken).
			SetHeader("Content-Type", "application/json").
			SetHeader("Accept", "application/json"),
	}
}

type Queue struct {
	Key string `json:"key,omitempty"`
}

type Parent struct {
	Key string `json:"key,omitempty"`
}

type Person struct {
	ID string `json:"id,omitempty"`
}

type Resolution struct {
	ID  string `json:"id,omitempty"`
	Key string `json:"key,omitempty"`
}

type Ticket struct {
	Description string      `json:"description,omitempty"`
	Queue       *Queue      `json:"queue,omitempty"`
	Summary     string      `json:"summary,omitempty"`
	Tags        []string    `json:"tags,omitempty"`
	Assignee    *Person     `json:"assignee,omitempty"`
	Followers   []Person    `json:"followers,omitempty"`
	Resolution  *Resolution `json:"resolution,omitempty"`
	MacroTag    string      `json:"macroTag,omitempty"`
	Parent      *Parent     `json:"parent,omitempty"`
}

type Transition struct {
	Resolution string `json:"resolution,omitempty"`
}

type TransitionItem struct {
	ID string       `json:"id,omitempty"`
	To TicketStatus `json:"to,omitempty"`
}

type TicketStatus struct {
	Key string `json:"key,omitempty"`
}

type TicketKey string

type NewTicketResponse struct {
	Key TicketKey `json:"key"`
}

type Comment struct {
	Text      string   `json:"text,omitempty"`
	Summonees []string `json:"summonees,omitempty"`
}

func (c *Client) CreateTicket(t *Ticket) (TicketKey, error) {
	resp, err := c.c.R().SetBody(t).SetResult(&NewTicketResponse{}).Post("/issues")
	if err != nil {
		return "", err
	}

	if resp.StatusCode() != http.StatusCreated {
		return "", fmt.Errorf("unsupported response code from service: %d: %s", resp.StatusCode(), resp.String())
	}

	return resp.Result().(*NewTicketResponse).Key, nil
}

func (c *Client) UpdateTicket(key TicketKey, t *Ticket, notify bool) error {
	resp, err := c.c.R().SetBody(t).Patch(fmt.Sprintf("/issues/%s?notify=%t", key, notify))
	if err != nil {
		return err
	}

	if resp.StatusCode() != http.StatusOK {
		return fmt.Errorf("unsupported response code from service: %d: %s", resp.StatusCode(), resp.String())
	}

	return nil
}

func (c *Client) CloseTicket(key TicketKey) error {
	var transitions []TransitionItem
	_, err := c.c.R().SetResult(&transitions).Get(fmt.Sprintf("/issues/%s/transitions", key))
	if err != nil {
		return err
	}

	closedTransitionID := ""
	for _, transition := range transitions {
		if transition.To.Key == "closed" {
			closedTransitionID = transition.ID
			break
		}
	}
	if closedTransitionID == "" {
		return fmt.Errorf("failed to close ticket: no closed transision found")
	}

	t := Transition{Resolution: "fixed"}
	resp, err := c.c.R().SetBody(t).Post(fmt.Sprintf("/issues/%s/transitions/%s/_execute", key, closedTransitionID))
	if err != nil {
		return err
	}

	if resp.StatusCode() != http.StatusOK {
		return fmt.Errorf("unsupported response code from service: %d: %s", resp.StatusCode(), resp.String())
	}

	return nil
}

func (c *Client) CommentTicket(key TicketKey, comment Comment) error {
	resp, err := c.c.R().SetBody(comment).Post(fmt.Sprintf("/issues/%s/comments", key))
	if err != nil {
		return err
	}

	if resp.StatusCode() != http.StatusCreated {
		return fmt.Errorf("unsupported response code from service: %d: %s", resp.StatusCode(), resp.String())
	}

	return nil
}

func (c *Client) IsTicketOpened(key TicketKey) (bool, error) {
	resp, err := c.c.R().SetResult(&Ticket{}).Get(fmt.Sprintf("/issues/%s/", key))
	if err != nil {
		return false, err
	}

	if resp.StatusCode() != http.StatusOK {
		return false, fmt.Errorf("unsupported response code from service: %d: %s", resp.StatusCode(), resp.String())
	}

	return resp.Result().(*Ticket).Resolution == nil, nil
}

type StatusID int

const (
	StatusOpen StatusID = 1
)

type Status struct {
	ID StatusID `json:"id"`
}

type Filter struct {
	Queue    string `json:"queue"`
	Status   Status `json:"status"`
	MacroTag string `json:"macroTag"`
}

type TicketsFilter struct {
	Filter Filter `json:"filter"`
}

func (c *Client) GetTickets(filter *TicketsFilter) ([]*Ticket, error) {
	response := []*Ticket{}
	rsp, err := c.c.R().SetResult(&response).SetBody(filter).Post("/issues/_search")
	if err != nil {
		return nil, fmt.Errorf("request failed: %w", err)
	}
	if rsp.StatusCode() != http.StatusOK {
		return nil, fmt.Errorf("unsupported response code from service: %d: %s", rsp.StatusCode(), rsp.String())
	}

	return response, nil
}
