package wiki

import (
	"fmt"
	"net/http"
	"strings"
	"time"

	"github.com/go-resty/resty/v2"
)

const (
	defaultURL            = "https://wiki-api.yandex-team.ru/_api/"
	defaultRequestTimeout = time.Second * 30
)

type ClientConfig struct {
	URL            string        `yaml:"url"`
	OauthToken     string        `yaml:"token"`
	RequestTimeout time.Duration `yaml:"request_timeout"`
}

type Client struct {
	c *resty.Client
}

func NewClient(config *ClientConfig) *Client {
	requestTimeout := config.RequestTimeout
	if requestTimeout == 0 {
		requestTimeout = defaultRequestTimeout
	}

	URL := config.URL
	if URL == "" {
		URL = defaultURL
	}

	return &Client{
		c: resty.New().
			SetTimeout(requestTimeout).
			SetHostURL(URL).
			SetAuthScheme("OAuth").
			SetAuthToken(config.OauthToken).
			SetHeader("Content-Type", "application/json").
			SetHeader("Accept", "application/json"),
	}
}

type Page struct {
	Body  string `json:"body,omitempty"`
	Title string `json:"title,omitempty"`
}

type PageResponse struct {
	Data *Page `json:"data,omitempty"`
}

func (c *Client) GetPage(supertag string) (*Page, error) {
	resp, err := c.c.R().SetResult(&PageResponse{}).Get(fmt.Sprintf("/frontend/%s/.raw", strings.Trim(supertag, "/")))
	if err != nil {
		return nil, err
	}

	if resp.StatusCode() != http.StatusOK {
		return nil, fmt.Errorf("unsupported response code from service: %d: %s", resp.StatusCode(), resp.String())
	}

	return resp.Result().(*PageResponse).Data, nil
}
