package yp

import (
	"a.yandex-team.ru/yp/go/yp"
	"context"
	"google.golang.org/protobuf/proto"
)

// implements ReleasesGetter, DeployTicketsGetter and other getters
type Client struct {
	base    *yp.Client
	cluster string
}

func NewClient(cluster string, token string) (*Client, error) {
	base, err := yp.NewClient(cluster, yp.WithAuthToken(token))
	if err != nil {
		return nil, err
	}
	return &Client{
		base:    base,
		cluster: cluster,
	}, nil
}

func (c *Client) Releases() ReleaseInterface {
	return newReleases(c)
}

func (c *Client) Stages() StageInterface {
	return newStages(c)
}

func (c *Client) DeployTickets() DeployTicketInterface {
	return newDeployTickets(c)
}

type batchInterface interface {
	MakeEmptyItem() proto.Message
	AddItem(m proto.Message) error
	SetContinuationToken(t string)
}

func selectAndFillBatch(ctx context.Context, c *yp.Client, filter string, objectType yp.ObjectType, selectors []string, batchSize int, continuationToken string, batch batchInterface) error {
	req := yp.SelectObjectsRequest{
		ObjectType:        objectType,
		Filter:            filter,
		Format:            yp.PayloadFormatProto,
		Selectors:         selectors,
		Limit:             int32(batchSize),
		ContinuationToken: continuationToken,
		FetchRootObject:   true,
	}
	rsp, err := c.SelectObjects(ctx, req)
	if err != nil {
		return err
	}
	for rsp.Next() {
		var r = batch.MakeEmptyItem()
		if err := rsp.Fill(r); err != nil {
			return err
		}
		if err := batch.AddItem(r); err != nil {
			return err
		}
	}
	batch.SetContinuationToken(rsp.ContinuationToken())
	return nil
}

func getAndFillBatchByIDs(ctx context.Context, c *yp.Client, IDs []string, objectType yp.ObjectType, selectors []string, batch batchInterface) error {
	req := yp.GetObjectsRequest{
		ObjectType:      objectType,
		ObjectIDs:       IDs,
		Format:          yp.PayloadFormatProto,
		Selectors:       selectors,
		FetchRootObject: true,
	}
	rsp, err := c.GetObjects(ctx, req)
	if err != nil {
		return err
	}
	for rsp.Next() {
		var r = batch.MakeEmptyItem()
		if err := rsp.Fill(r); err != nil {
			return err
		}
		if err := batch.AddItem(r); err != nil {
			return err
		}
	}
	return nil
}

func removeByIDs(ctx context.Context, c *yp.Client, IDs []string, objectType yp.ObjectType) error {
	objs := make([]yp.RemoveObject, 0, len(IDs))
	for _, ID := range IDs {
		objs = append(objs, yp.RemoveObject{
			ObjectType: objectType,
			ObjectID:   ID,
		})
	}
	req := yp.RemoveObjectsRequest{
		Objects:           objs,
		IgnoreNonexistent: true,
	}
	_, err := c.RemoveObjects(ctx, req)
	return err
}
