package main

import (
	"a.yandex-team.ru/infra/temporal/clients/startrek"
	"a.yandex-team.ru/infra/temporal/workflows/startreker/processor"
	"context"
	"encoding/json"
	"fmt"
	"github.com/spf13/cobra"
	workflowpb "go.temporal.io/api/workflow/v1"
	"go.temporal.io/api/workflowservice/v1"
	"go.temporal.io/sdk/client"
	"go.uber.org/zap"
	"io/ioutil"
	"sort"
	"strings"
)

func recreateStartrekerXFFYTickets() *cobra.Command {
	cmd := &cobra.Command{
		Use: "recreate-startreker-xffy-tickets",
		RunE: func(cmd *cobra.Command, args []string) error {
			c, err := createTemporalClient()
			if err != nil {
				return err
			}
			defer c.Close()

			input, err := cmd.Flags().GetString("input")
			if err != nil {
				return err
			}

			content, err := ioutil.ReadFile(input)
			if err != nil {
				return err
			}

			var result map[string]processor.Problem
			_ = json.Unmarshal([]byte(content), &result)

			for key, problem := range result {
				items := strings.Split(key, "/")
				wfID := items[0]
				startrekID := startrek.TicketKey(items[1])
				options := client.StartWorkflowOptions{
					ID:        wfID,
					TaskQueue: "startreker",
				}
				we, err := c.ExecuteWorkflow(context.Background(), options, processor.ContinueProcessingWorkflow, startrekID, problem)
				if err != nil {
					return fmt.Errorf("failed to start ProcessWorkflow workflow: %w", err)
				}
				logger.Info("started ContinueProcessingWorkflow workflow",
					zap.String("ID", we.GetID()), zap.String("RunID", we.GetRunID()))
				break
			}

			return nil
		},
	}
	cmd.Flags().String("input", "", "AWACS-1238 input data")
	_ = cmd.MarkFlagRequired("input")
	return cmd
}

func listAllNamespaces(ctx context.Context, c workflowservice.WorkflowServiceClient) ([]*workflowservice.DescribeNamespaceResponse, error) {
	var res []*workflowservice.DescribeNamespaceResponse
	pagesize := int32(1000)
	var token []byte
	ctx, cancel := context.WithCancel(ctx)
	defer cancel()
	for more := true; more; more = len(token) > 0 {
		listRequest := &workflowservice.ListNamespacesRequest{
			PageSize:      pagesize,
			NextPageToken: token,
		}
		listResp, err := c.ListNamespaces(ctx, listRequest)
		if err != nil {
			return nil, err
		}
		token = listResp.GetNextPageToken()
		res = append(res, listResp.GetNamespaces()...)
	}
	return res, nil
}

func listOpenWorkflowExecutions(ctx context.Context, c workflowservice.WorkflowServiceClient, namespace string) ([]*workflowpb.WorkflowExecutionInfo, error) {
	var res []*workflowpb.WorkflowExecutionInfo

	pagesize := int32(1000)
	var token []byte
	ctx, cancel := context.WithCancel(ctx)
	defer cancel()
	for more := true; more; more = len(token) > 0 {
		listRequest := &workflowservice.ListOpenWorkflowExecutionsRequest{
			MaximumPageSize: pagesize,
			NextPageToken:   token,
			Namespace:       namespace,
		}
		listResp, err := c.ListOpenWorkflowExecutions(ctx, listRequest)
		if err != nil {
			return nil, err
		}
		token = listResp.GetNextPageToken()
		res = append(res, listResp.GetExecutions()...)
	}
	return res, nil
}

func listAllNamespaceNames(ctx context.Context, c workflowservice.WorkflowServiceClient) ([]string, error) {
	namespaceDescs, err := listAllNamespaces(ctx, c)
	if err != nil {
		return nil, err
	}
	var res []string
	for _, desc := range namespaceDescs {
		res = append(res, desc.NamespaceInfo.Name)
	}
	return res, nil
}

func listAllStartedWorkflows() *cobra.Command {
	cmd := &cobra.Command{
		Use: "list-all-started-workflows",
		RunE: func(cmd *cobra.Command, args []string) error {
			c, err := createTemporalFrontendClient()
			if err != nil {
				return err
			}
			namespaces, err := listAllNamespaceNames(cmd.Context(), c)
			if err != nil {
				return err
			}
			sort.Strings(namespaces)
			for _, namespace := range namespaces {
				executions, _ := listOpenWorkflowExecutions(cmd.Context(), c, namespace)
				fmt.Printf("%s (%d open wfs):\n", namespace, len(executions))
				var wfIDs []string
				for _, e := range executions {
					wfIDs = append(wfIDs, e.Execution.WorkflowId)
				}
				sort.Strings(wfIDs)
				for _, wfID := range wfIDs {
					fmt.Printf("* %s\n", wfID)
				}
			}
			return nil
		},
	}
	return cmd
}

func init() {
	rootCmd.AddCommand(recreateStartrekerXFFYTickets())
	rootCmd.AddCommand(listAllStartedWorkflows())
}
