package main

import (
	"a.yandex-team.ru/library/go/yandex/oauth"
	"context"
	"fmt"
	"github.com/spf13/cobra"
	"go.uber.org/zap"
	"go.uber.org/zap/zapcore"
	"math/rand"
	"os"
	"time"
)

var logger *zap.Logger
var rootCmd = &cobra.Command{
	Use:   "atemporal",
	Short: "Perform operations in awacs universe through Temporal",
	Long: `Authorization:
The app tries to get awacs token using one of three ways in the following order:
* read AWACS_TOKEN env var
* exchange ssh key
To get a token manually click on the lock icon here: https://nanny.yandex-team.ru/ui/#/awacs/
`,
	Run: func(cmd *cobra.Command, args []string) {
	},
}

const (
	TokenEnvVarName             = "AWACS_TOKEN"
	TemporalNamespaceEnvVarName = "TEMPORAL_NAMESPACE"
)

func setAwacsToken() {
	if os.Getenv(TokenEnvVarName) != "" {
		logger.Info("got token from env var", zap.String("name", TokenEnvVarName))
		return
	}
	set := func(v string) {
		if err := os.Setenv(TokenEnvVarName, v); err != nil {
			logger.Error("unable to set env var", zap.String("name", TokenEnvVarName), zap.Error(err))
		}
	}
	token, errSSH := getOAuthToken(context.Background())
	if errSSH == nil {
		logger.Info("exchanged ssh key for oauth token")
		set(token)
		return
	}
	logger.Error("unable to exchange ssh key for oauth token", zap.Error(errSSH))
}

const (
	OAuthClientID     = "447a6c29a65345f0a4d31784bb850bbd"
	OAuthClientSecret = "ff95f1663807472a91f8a3757a68586b"
)

// getOAuthToken exchanges ssh key for awacs oauth token.
func getOAuthToken(ctx context.Context) (string, error) {
	token, err := oauth.GetTokenBySSH(ctx, OAuthClientID, OAuthClientSecret)
	if err != nil {
		return "", fmt.Errorf("error exchanging ssh key for oauth token: %w", err)
	}
	return token, nil
}

func init() {
	rand.Seed(int64(time.Now().Nanosecond()))
	logger, _ = zap.NewDevelopment(
		zap.AddStacktrace(zapcore.FatalLevel),
		zap.AddCallerSkip(1),
	)
	cobra.OnInitialize(setAwacsToken)
}

func main() {
	if err := rootCmd.Execute(); err != nil {
		fmt.Fprintln(os.Stderr, err)
		os.Exit(1)
	}
}
