package main

import (
	"a.yandex-team.ru/infra/temporal/clients/staff"
	"a.yandex-team.ru/infra/temporal/clients/wiki"
	temporalclient "a.yandex-team.ru/infra/temporal/swat/client"
	"flag"
	"fmt"
	prometheus "github.com/uber-go/tally/v4/prometheus"
	"go.uber.org/zap"
	"log"
	"os"

	"go.temporal.io/sdk/worker"

	awacsactivities "a.yandex-team.ru/infra/temporal/activities/awacs"
	calendaractivities "a.yandex-team.ru/infra/temporal/activities/calendar"
	swatactivities "a.yandex-team.ru/infra/temporal/activities/swat"
	awacsworkflows "a.yandex-team.ru/infra/temporal/workflows/awacs"
	samplesworkflows "a.yandex-team.ru/infra/temporal/workflows/samples"
)

const (
	AwacsTokenEnvVarName   = "AWACS_TOKEN"
	StaffTokenEnvVarName   = "STAFF_TOKEN"
	WikiTokenEnvVarName    = "WIKI_TOKEN"
	AwacsAPIURLEnvVarName  = "AWACS_API_URL"
	FrontAddr              = "swat-temporal.yandex.net:7232"
	AwacsTaskQueue         = "awacs"
	AwacsWatchingTaskQueue = "awacs-watching"
)

var workflows = []interface{}{
	samplesworkflows.PoolMultiParallelChildWorkflow,
	samplesworkflows.SleepingWorkflow,

	awacsworkflows.ContinueRenewedCertificateDeployment,
	awacsworkflows.DeployCertificateRenewal,
	awacsworkflows.DummyWorkflow,
	awacsworkflows.MigrateBackendToSDWorkflow,
	awacsworkflows.ScheduleCertificateRenewalDeployments,
	awacsworkflows.CronNamespacesWorkflow,
	awacsworkflows.CreateNamespaceWorkflowsWorkflow,
}

var watchingWorkflows = []interface{}{
	awacsworkflows.ProcessNamespaceWorkflow,
}

var logger *zap.Logger

func main() {
	var namespace string
	flag.StringVar(&namespace, "n", "", "Specify temporal namespace ID")
	flag.Parse()
	if namespace == "" {
		log.Fatal("namespace must be specified")
	}

	token := os.Getenv("TEMPORAL_TOKEN")
	if token == "" {
		log.Fatalln("TEMPORAL_TOKEN must be set")
	}

	scope, err := temporalclient.NewPrometheusScope(prometheus.Configuration{
		ListenAddress: "127.0.0.1:31337",
		TimerType:     "histogram",
	}, logger)
	if err != nil {
		log.Fatalln(fmt.Errorf("unable to create Prometheus scope: %w", err))
	}

	c, err := temporalclient.NewSdkClient(FrontAddr, token, namespace, temporalclient.WithMetricsScope(scope))
	if err != nil {
		log.Fatalln("failed to create temporal client", err)
	}
	defer c.Close()

	w := worker.New(c, AwacsTaskQueue, worker.Options{})
	for _, wf := range workflows {
		w.RegisterWorkflow(wf)
	}
	awacsURL := os.Getenv(AwacsAPIURLEnvVarName)
	awacsToken := os.Getenv(AwacsTokenEnvVarName)
	staffClientConfig := &staff.ClientConfig{
		OauthToken: os.Getenv(StaffTokenEnvVarName),
	}
	activities := awacsactivities.NewActivities(awacsURL, awacsToken, staffClientConfig)
	swatActivities := swatactivities.NewActivities(&wiki.ClientConfig{OauthToken: os.Getenv(WikiTokenEnvVarName)})

	w.RegisterActivity(swatActivities)
	w.RegisterActivity(activities)
	w.RegisterActivity(calendaractivities.IsNowWorkingHours)
	w.RegisterActivity(calendaractivities.IsTodayWorkingDay)

	watchingActivities := awacsactivities.NewWatchingActivities(awacsURL, awacsToken, staffClientConfig)
	watchingW := worker.New(c, AwacsWatchingTaskQueue, worker.Options{
		MaxConcurrentActivityExecutionSize:     10000,
		MaxConcurrentWorkflowTaskExecutionSize: 5000,
		MaxConcurrentActivityTaskPollers:       6,
		MaxConcurrentWorkflowTaskPollers:       3,
	})
	for _, wf := range watchingWorkflows {
		watchingW.RegisterWorkflow(wf)
	}
	watchingW.RegisterActivity(watchingActivities)
	watchingW.RegisterActivity(swatActivities)
	watchingW.RegisterActivity(activities)

	errs := make(chan error)
	go func() {
		var e error
		defer func() { errs <- e }()
		e = w.Run(worker.InterruptCh())
	}()
	go func() {
		var e error
		defer func() { errs <- e }()
		e = watchingW.Run(worker.InterruptCh())
	}()

	err = <-errs
	if err != nil {
		log.Fatalln("failed to start", err)
	}

	err = <-errs
	if err != nil {
		log.Fatalln("failed to start", err)
	}
}
