package main

import (
	"context"
	"flag"
	"fmt"
	"log"
	"os"
	"regexp"
	"strings"

	"go.temporal.io/sdk/client"

	pb "a.yandex-team.ru/infra/nanny/go/proto/nanny_repo"
	"a.yandex-team.ru/infra/nanny2/pkg/rpc"
	"a.yandex-team.ru/infra/temporal/activities/nanny/services"
	temporalClient "a.yandex-team.ru/infra/temporal/swat/client"
	podAlerts "a.yandex-team.ru/infra/temporal/workflows/nanny/podalerts"
)

const TemporalTokenEnvVarName = "TEMPORAL_TOKEN"
const NannyTokenEnvVarName = "NANNY_TOKEN"

var Regexps = []string{
	"^saas_",
}

func validateRegexps() {
	for _, r := range Regexps {
		regexp.MustCompile(r)
	}
}

func handleDryRun(fileName string) {
	token := os.Getenv(NannyTokenEnvVarName)
	if token == "" {
		log.Fatalf("NANNY_TOKEN must be set to dry-run")
	}

	nannyURL := "https://nanny.yandex-team.ru"
	rpcConfig := rpc.ClientConfig{
		RPCURL:     fmt.Sprintf("%s/api/repo", nannyURL),
		OauthToken: token,
	}
	rpcClient := pb.NewRepoServiceClient(rpc.NewClient(&rpcConfig))

	a := services.NewActivities(nannyURL, token, rpcClient)
	r := services.GetServicesRequest{
		Skip:         0,
		Limit:        30000,
		Regexps:      Regexps,
		DeployEngine: "YP_LITE",
	}
	services, err := a.GetServicesActivity(context.Background(), &r)
	if err != nil {
		log.Fatalf(err.Error())
	}

	var ypServices []string
	for _, service := range services {
		ypServices = append(ypServices, service.ID)
	}
	servicesBytes := []byte(strings.Join(ypServices, "\n"))

	fd, err := os.Create(fileName)
	if err != nil {
		log.Fatalf(err.Error())
	}
	defer fd.Close()

	_, err = fd.Write(servicesBytes)
	if err != nil {
		log.Fatalf(err.Error())
	}
	fmt.Printf("Detected %d services\n", len(services))
}

func main() {
	namespace := flag.String("ns", "", "temporal namespace")
	taskQueue := flag.String("tq", "", "temporal task queue")
	dryRun := flag.String("dry-run", "", "if present, only outputs services in this file")
	flag.Parse()
	if *namespace == "" || *taskQueue == "" {
		flag.Usage()
		os.Exit(1)
	}

	validateRegexps()

	if *dryRun != "" {
		handleDryRun(*dryRun)
		return
	}

	c, err := temporalClient.NewSdkClient(
		"swat-temporal.yandex.net:7232",
		os.Getenv(TemporalTokenEnvVarName),
		*namespace)
	if err != nil {
		log.Fatalln("unable to create Temporal client", err)
	}
	defer c.Close()

	options := client.StartWorkflowOptions{
		ID:           "CronPodAlertsWorkflow",
		TaskQueue:    *taskQueue,
		CronSchedule: "*/30 * * * *", // every 30 minutes
	}

	wr, err := c.ExecuteWorkflow(
		context.Background(),
		options,
		podAlerts.CronPodAlertsWorkflow,
		&podAlerts.Config{},
		&podAlerts.PodAlertsWorkflowState{
			Offset:        0,
			ServicesCount: 0,
		})
	if err != nil {
		log.Fatalln("error starting workflow:", err)
	}
	log.Printf(
		"Finished!\nWorkflowID: %s, RunID: %s\n",
		wr.GetID(),
		wr.GetRunID())
}
