package main

import (
	"errors"
	"flag"
	"fmt"
	"log"
	"os"
	"strings"

	"go.temporal.io/sdk/worker"

	pb "a.yandex-team.ru/infra/nanny/go/proto/nanny_repo"
	"a.yandex-team.ru/infra/nanny2/pkg/rpc"
	"a.yandex-team.ru/infra/temporal/activities/abc"
	"a.yandex-team.ru/infra/temporal/activities/io"
	"a.yandex-team.ru/infra/temporal/activities/nanny/pods"
	getServices "a.yandex-team.ru/infra/temporal/activities/nanny/services"
	abcClient "a.yandex-team.ru/infra/temporal/clients/abc"
	temporalClient "a.yandex-team.ru/infra/temporal/swat/client"
	podAlerts "a.yandex-team.ru/infra/temporal/workflows/nanny/podalerts"
	servicePodAlerts "a.yandex-team.ru/infra/temporal/workflows/nanny/servicepodalerts"
)

func getEnv(name string) string {
	value := os.Getenv(name)
	if value == "" {
		log.Fatalf("%s must be set\n", name)
	}
	return value
}

func validateConfig(envName string) {
	fileName, exists := os.LookupEnv(envName)
	if !exists {
		log.Fatalf("%s is not set", envName)
	}

	if _, err := os.Stat(fileName); errors.Is(err, os.ErrNotExist) {
		log.Fatalf("config file for %s does not exist", envName)
	}
}

func main() {
	temporalToken := getEnv("TEMPORAL_TOKEN")
	temporalURL := getEnv("TEMPORAL_URL")
	nannyToken := getEnv("NANNY_TOKEN")
	nannyURL := getEnv("NANNY_API_URL")
	ypToken := getEnv("YP_TOKEN")
	abcToken := getEnv("ABC_TOKEN")
	ypClusters := os.Getenv("YP_CLUSTERS")

	taskQueue := flag.String("tq", "", "taskQueue")
	namespace := flag.String("ns", "", "namespace")
	flag.Parse()
	if *taskQueue == "" || *namespace == "" {
		flag.Usage()
		os.Exit(1)
	}

	validateConfig(podAlerts.ConfigFileNameEnvironmentVariable)
	validateConfig(servicePodAlerts.ConfigFileNameEnvironmentVariable)

	client, err := temporalClient.NewSdkClient(temporalURL, temporalToken, *namespace)
	if err != nil {
		log.Fatalln("unable to create temporal client: ", err)
	}
	defer client.Close()

	w := worker.New(client, *taskQueue, worker.Options{})

	w.RegisterWorkflow(podAlerts.PodAlertsWorkflow)
	w.RegisterWorkflow(podAlerts.CronPodAlertsWorkflow)
	w.RegisterWorkflow(servicePodAlerts.ServicePodAlertsWorkflow)

	rpcConfig := rpc.ClientConfig{
		RPCURL:     fmt.Sprintf("%s/api/repo", nannyURL),
		OauthToken: nannyToken,
	}
	rpcClient := pb.NewRepoServiceClient(rpc.NewClient(&rpcConfig))
	servicesActivities := getServices.NewActivities(nannyURL, nannyToken, rpcClient)

	clusters := []string{"sas", "man", "vla", "iva", "myt"}
	if ypClusters != "" {
		clusters = strings.Split(ypClusters, ",")
	}
	podsActivities, err := pods.NewActivities(ypToken, clusters)
	if err != nil {
		log.Fatalln("unable to start pod activities", err)
	}
	abcActivities := abc.NewActivities(&abcClient.ClientConfig{
		OauthToken: abcToken,
	})

	w.RegisterActivity(servicesActivities)
	w.RegisterActivity(podsActivities)
	w.RegisterActivity(abcActivities)
	w.RegisterActivity(&io.Activities{})

	err = w.Run(worker.InterruptCh())
	if err != nil {
		log.Fatalln("unable to start worker", err)
	}
}
