package main

import (
	"context"
	"flag"
	"log"
	"net/http"
	"os"
	"time"

	"a.yandex-team.ru/infra/temporal/clients/abc"
	"a.yandex-team.ru/infra/temporal/clients/startrek"
	temporalclient "a.yandex-team.ru/infra/temporal/swat/client"

	"go.temporal.io/sdk/worker"

	calendaractivities "a.yandex-team.ru/infra/temporal/activities/calendar"
	"a.yandex-team.ru/infra/temporal/activities/startreker"
	"a.yandex-team.ru/infra/temporal/workflows/startreker/processor"
)

const (
	startrekOauthTokenEnvVar = "STARTREK_TOKEN"
	awacsOauthTokenEnvVar    = "ABC_TOKEN"
)

func main() {
	var namespace string
	var listener string
	flag.StringVar(&namespace, "n", "", "Specify temporal namespace ID")
	flag.StringVar(&listener, "l", ":80", "Specify listener for http server")
	flag.Parse()
	if namespace == "" {
		log.Fatal("namespace must be specified")
	}

	c, err := temporalclient.NewSdkClient(
		"swat-temporal.yandex.net:7232",
		os.Getenv("TEMPORAL_TOKEN"),
		namespace)
	if err != nil {
		log.Fatalln("unable to create Temporal client", err)
	}
	defer c.Close()

	w := worker.New(c, "startreker", worker.Options{})
	w.RegisterWorkflow(processor.ProcessWorkflow)
	w.RegisterWorkflow(processor.ContinueProcessingWorkflow)

	startrekOauthToken := os.Getenv(startrekOauthTokenEnvVar)
	if startrekOauthToken == "" {
		log.Fatalf("unable to get startrek OAuth token from env var %s\n", startrekOauthTokenEnvVar)
	}
	awacsOauthToken := os.Getenv(awacsOauthTokenEnvVar)
	if startrekOauthToken == "" {
		log.Fatalf("unable to get awacs OAuth token from env var %s\n", awacsOauthTokenEnvVar)
	}

	activities := startreker.NewActivities(
		startrek.ClientConfig{
			OauthToken: startrekOauthToken,
		},
		abc.ClientConfig{
			OauthToken: awacsOauthToken,
		},
	)
	w.RegisterActivity(activities)
	w.RegisterActivity(calendaractivities.IsNowWorkingHours)
	w.RegisterActivity(calendaractivities.IsTodayWorkingDay)

	httpSrv := http.Server{
		Addr:         listener,
		Handler:      NewRouter(context.TODO(), c),
		ReadTimeout:  time.Second * 30,
		WriteTimeout: time.Second * 30,
	}
	go httpSrv.ListenAndServe()

	err = w.Run(worker.InterruptCh())
	if err != nil {
		log.Fatalln("unable to start Worker", err)
	}
}
