package main

import (
	temporalClient "a.yandex-team.ru/infra/temporal/swat/client"
	wfs "a.yandex-team.ru/infra/temporal/workflows/yd/ri/gc"
	"context"
	"flag"
	"go.temporal.io/sdk/client"
	"log"
	"os"
	"time"
)

const TemporalTokenEnvVarName = "TEMPORAL_TOKEN"

func main() {
	namespace := flag.String("ns", "", "temporal namespace")
	taskQueue := flag.String("tq", "", "temporal task queue")
	gcObjType := flag.String("gc", "", "object type to gc, must be one of release or ticket")

	flag.Parse()
	if *namespace == "" || *taskQueue == "" {
		flag.Usage()
		os.Exit(1)
	}

	var wf interface{}
	var wfID string
	var cron string
	switch *gcObjType {
	case "ticket":
		wf = wfs.CronDeployTicketsGCWorkflow
		wfID = "CronDeployTicketsGCWorkflow"
		cron = "0 13 * * 2" // every Tuesday at 1 pm
	case "release":
		wf = wfs.CronReleasesGCWorkflow
		wfID = "CronReleasesGCWorkflow"
		cron = "0 13 * * 4" // every Thursday at 1 pm
	default:
		flag.Usage()
		os.Exit(1)
	}

	c, err := temporalClient.NewSdkClient(
		"swat-temporal.yandex.net:7232",
		os.Getenv(TemporalTokenEnvVarName),
		*namespace)
	if err != nil {
		log.Fatalln("unable to create Temporal client", err)
	}

	options := client.StartWorkflowOptions{
		ID:           wfID,
		TaskQueue:    *taskQueue,
		CronSchedule: cron,
	}

	wr, err := c.ExecuteWorkflow(
		context.Background(),
		options,
		wf,
		&wfs.Config{
			DeployTicketsBatchSize:             500,
			DeployTicketsBatchesCountInPortion: 20,
			DeployTicketsRemoveBatchSize:       50,
			ReleasesBatchSize:                  500,
			ReleasesBatchesCountInPortion:      20,
			ReleasesRemoveBatchSize:            50,
			StagesBatchSize:                    500,
			SOXDeployTicketTTL:                 time.Hour * 24 * 400,
			DeployTicketTTL:                    time.Hour * 24 * 200,
			ReleaseTTL:                         time.Hour * 24 * 200,
		},
	)

	if err != nil {
		log.Fatalf("error starting workflow %s-gc: %s", *gcObjType, err)
	}

	log.Printf(
		"Finished!\nWorkflowID: %s, RunID: %s\n",
		wr.GetID(),
		wr.GetRunID())
}
