package main

import (
	"flag"
	"log"
	"os"

	"go.temporal.io/sdk/worker"

	acts "a.yandex-team.ru/infra/temporal/activities/yd/ri"
	temporalClient "a.yandex-team.ru/infra/temporal/swat/client"
	"a.yandex-team.ru/infra/temporal/workflows/yd/ri/gc"
)

func getEnv(name string) string {
	value := os.Getenv(name)
	if value == "" {
		log.Fatalf("%s must be set\n", name)
	}
	return value
}

func main() {
	temporalToken := getEnv("TEMPORAL_TOKEN")
	temporalURL := getEnv("TEMPORAL_URL")
	ypToken := getEnv("YP_TOKEN")
	ypCluster := os.Getenv("YP_CLUSTER")

	taskQueue := flag.String("tq", "", "taskQueue")
	namespace := flag.String("ns", "", "namespace")

	flag.Parse()
	if *taskQueue == "" || *namespace == "" {
		flag.Usage()
		os.Exit(1)
	}

	client, err := temporalClient.NewSdkClient(temporalURL, temporalToken, *namespace)
	if err != nil {
		log.Fatalln("unable to create temporal client: ", err)
	}
	defer client.Close()

	w := worker.New(client, *taskQueue, worker.Options{})

	w.RegisterWorkflow(gc.DeployTicketsPortionGCWorkflow)
	w.RegisterWorkflow(gc.CronDeployTicketsGCWorkflow)
	w.RegisterWorkflow(gc.ReleasesPortionGCWorkflow)
	w.RegisterWorkflow(gc.CronReleasesGCWorkflow)

	a, err := acts.NewActivities(ypCluster, ypToken)
	if err != nil {
		log.Fatalln("unable to create YP activity", err)
	}
	w.RegisterActivity(a)
	err = w.Run(worker.InterruptCh())
	if err != nil {
		log.Fatalln("unable to start worker", err)
	}
}
