import json
import os
import requests
from google.protobuf.json_format import MessageToDict, Parse
from temporal.api.enums.v1 import event_type_pb2
from temporal.api.workflowservice.v1 import request_response_pb2

DEFAULT_AWACS_RPC_URL = 'http://swat-temporal.yandex.net:8080'


def req(method, req_pb, resp_pb):
    url = DEFAULT_AWACS_RPC_URL + '/temporal.api.workflowservice.v1.WorkflowService/{}'.format(method)
    content = requests.post(url, json=MessageToDict(req_pb), headers={
        'Authorization': os.getenv('TEMPORAL_TOKEN'),
    }).content
    Parse(content, resp_pb)


def list_open_wf_executions(namespace):
    next_page_token = b''
    while 1:
        req_pb = request_response_pb2.ListOpenWorkflowExecutionsRequest(
            namespace=namespace,
            maximum_page_size=100,
            next_page_token=next_page_token
        )
        resp_pb = request_response_pb2.ListOpenWorkflowExecutionsResponse()
        req('ListOpenWorkflowExecutions', req_pb, resp_pb)
        for execution_pb in resp_pb.executions:
            yield execution_pb
        next_page_token = resp_pb.next_page_token
        if not next_page_token:
            return


def iter_wf_exec_history(namespace, execution_pb):
    next_page_token = b''
    while 1:
        req_pb = request_response_pb2.GetWorkflowExecutionHistoryRequest(
            namespace=namespace,
            execution=execution_pb,
            next_page_token=next_page_token
        )
        resp_pb = request_response_pb2.GetWorkflowExecutionHistoryResponse()
        req('GetWorkflowExecutionHistory', req_pb, resp_pb)
        for event_pb in resp_pb.history.events:
            yield event_pb
        next_page_token = resp_pb.next_page_token
        if not next_page_token:
            return


if __name__ == '__main__':
    ns = 'startreker'
    rv = {}
    for execution_pb in list_open_wf_executions(ns):
        name = execution_pb.type.name
        wf_id = execution_pb.execution.workflow_id
        if name == 'ProcessWorkflow' and wf_id.startswith('xffy'):
            event_pbs = list(iter_wf_exec_history(ns, execution_pb.execution))

            ticket_id = None
            for event_pb in event_pbs:
                if (
                    event_pb.event_type == event_type_pb2.EVENT_TYPE_ACTIVITY_TASK_SCHEDULED and
                    event_pb.activity_task_scheduled_event_attributes.activity_type.name == 'WaitUntilTicketClosed'
                ):
                    ticket_id = json.loads(event_pb.activity_task_scheduled_event_attributes.input.payloads[0].data)
                    break
            assert ticket_id and ticket_id.startswith('AWACSNOTIFY')

            first_event_pb = event_pbs[0]
            rv[wf_id + '/' + ticket_id] = json.loads(
                first_event_pb.workflow_execution_started_event_attributes.input.payloads[0].data)
    print(json.dumps(rv, indent=4))
