import requests
import click


class GrafanaAPI(object):
    def __init__(self, host, token):
        self.host = host.rstrip('/')
        self.token = token
        self.headers = {
            'Authorization': 'OAuth {}'.format(self.token),
            'Content-Type': 'application/json'
        }

    def post_dashboard(self, dashboard):
        url = '{}/api/dashboards/db'.format(self.host)
        result = requests.post(url, json=dashboard, headers=self.headers)
        result.raise_for_status()
        data = result.json()
        return "{}{}".format(self.host, data['url'])

    def get_dashboard(self, id):
        result = requests.request('GET', '{}/api/dashboards/uid/{}'.format(self.host, id),
                                  headers=self.headers)
        result.raise_for_status()
        return result.json()


def update_dashboard(dashboard_uid, dashboard, token):
    grafana_api = GrafanaAPI("https://grafana.yandex-team.ru", token)
    curr = grafana_api.get_dashboard(dashboard_uid)

    dashboard['uid'] = dashboard_uid
    dashboard['version'] = curr['meta']['version']
    dashboard['title'] = curr['dashboard']['title']
    dashboard['id'] = curr['dashboard']['id']

    url = grafana_api.post_dashboard({
        'dashboard': dashboard,
        'overwrite': True,
        'folderId': curr['meta']['folderId'],
        'message': 'infra/temporal/swat/junk',
    })
    click.secho('Take a look: ' + url, fg='green')


def replace_queries(dashboard, subs, indent='', verbose=True):
    """
    :param dict dashboard: dict with grafana dashboard
    :param dict subs: dict that maps PromQL queries to Solomon queries
    """
    for panel in dashboard['panels']:
        if verbose:
            click.echo(indent + '# ' + panel['title'])
        fixed = False
        for target in panel.get('targets', []):
            promql = target['expr']
            if promql in subs:
                solomonql = subs[promql]
                del target['expr']
                target['type'] = 'timeserie'
                target['target'] = solomonql
                if verbose:
                    click.secho(indent + 'PromQL:    ' + promql, fg='green')
                    click.secho(indent + 'SolomonQL: ' + solomonql)
                # XXX
                if target.get('legendFormat'):
                    target['refId'] = target['legendFormat']
                    del target['legendFormat']
                fixed = True
            elif promql:
                if verbose:
                    click.secho(indent + 'PromQL:   ' + promql, fg='red')
                    click.secho(indent + 'SolomonQL: -', fg='red')
        if fixed:
            panel['datasource'] = 'Solomon'
            panel['pluginVersion'] = '7.5.3'

        if 'panels' in panel:
            replace_queries(panel, subs, indent=indent + '  ')

        if verbose:
            click.echo()
