package main

import (
	"a.yandex-team.ru/library/go/slices"
	"a.yandex-team.ru/library/go/yandex/blackbox/httpbb"
	"context"
	"errors"
	"fmt"
	"github.com/karlseguin/ccache/v2"
	"go.temporal.io/server/common/authorization"
	"go.temporal.io/server/common/log"
	"go.temporal.io/server/common/log/tag"
	"google.golang.org/grpc/metadata"
	"google.golang.org/grpc/peer"
	"net"
	"strings"
)

type swatAuthorizer struct {
	bb     *httpbb.Client
	cache  *ccache.Cache
	logger log.Logger
	acl    map[string][]string
}

func NewSwatAuthorizer(logger log.Logger, acl map[string][]string) (authorization.Authorizer, error) {
	bb, err := httpbb.NewIntranet()
	if err != nil {
		return nil, fmt.Errorf("failed to create blackbox client: %w", err)
	}
	return &swatAuthorizer{
		bb,
		ccache.New(ccache.Configure().MaxSize(10000)),
		logger,
		acl,
	}, nil
}

var decisionAllow = authorization.Result{Decision: authorization.DecisionAllow}
var decisionDeny = authorization.Result{Decision: authorization.DecisionDeny}

const UnknownPeerIP = "0.0.0.0"

func getPeerIP(ctx context.Context) string {
	// Based on https://stackoverflow.com/questions/51753461/how-can-i-get-the-client-ip-address-and-user-agent-in-golang-grpc
	peerIP := UnknownPeerIP

	if p, ok := peer.FromContext(ctx); ok {
		host, _, err := net.SplitHostPort(p.Addr.String())
		if err == nil {
			peerIP = host
		}
	}
	if headers, ok := metadata.FromIncomingContext(ctx); ok {
		xForwardedFor := headers.Get("x-forwarded-for")
		if len(xForwardedFor) > 0 && xForwardedFor[0] != "" {
			ips := strings.Split(xForwardedFor[0], ",")
			if len(ips) > 0 {
				peerIP = ips[0]
			}
		}
	}

	return peerIP
}

func (a *swatAuthorizer) Authorize(ctx context.Context, claims *authorization.Claims, target *authorization.CallTarget) (authorization.Result, error) {
	if target.Namespace == "temporal-system" || target.Namespace == "" {
		// https://github.com/temporalio/temporal/blob/a87bf44cb796c472fc2943e9da7affded4ba3e6e/common/authorization/default_authorizer.go#L49-L56
		return decisionAllow, nil
	}

	peerIP := getPeerIP(ctx)
	peerIPTag := tag.NewStringTag("peerIP", peerIP)
	if claims == nil {
		a.logger.Warn("no claims", peerIPTag)
		return decisionDeny, nil
	}

	if claims.Subject == "swat-temporal-web" {
		// TODO @romanovich: switch web to readonly
		a.logger.Debug("request from swat-temporal-web", peerIPTag)
		return decisionAllow, nil
	}

	authToken, ok := claims.Extensions.(string)
	if !ok {
		return decisionDeny, errors.New("token is missing from claims")
	}

	login, cached, err := getLoginByOauthTokenWithCache(ctx, a.cache, a.bb, authToken, peerIP)
	if err != nil {
		a.logger.Warn("failed to call getLoginByOauthTokenWithCache", tag.Error(err), peerIPTag)
		return decisionDeny, err
	}

	allow := false
	roots := a.acl[""]
	if ok, _ := slices.Contains(roots, login); ok {
		allow = true
	} else if namespaceRoots, ok := a.acl[target.Namespace]; ok {
		allow, _ = slices.Contains(namespaceRoots, login)
	} else if target.Namespace == "user-"+login {
		allow = true
	}
	a.logger.Debug(fmt.Sprintf("request from %s@, oauth cached: %t, allowed: %t", login, cached, allow), peerIPTag)
	if allow {
		return decisionAllow, nil
	} else {
		return decisionDeny, nil
	}
}
