package main

import (
	"fmt"
	"github.com/urfave/cli/v2"
	"io/ioutil"
	"os"
	"path"

	"go.temporal.io/server/common/config"
	"go.temporal.io/server/common/headers"
	tlog "go.temporal.io/server/common/log"
	_ "go.temporal.io/server/common/persistence/sql/sqlplugin/mysql"      // needed to load mysql plugin
	_ "go.temporal.io/server/common/persistence/sql/sqlplugin/postgresql" // needed to load postgresql plugin
	"go.temporal.io/server/temporal"
	"gopkg.in/yaml.v2"
)

func main() {
	app := buildCLI()
	_ = app.Run(os.Args)
}

func buildCLI() *cli.App {
	app := cli.NewApp()
	app.Name = "temporal"
	app.Usage = "Temporal server"
	app.Version = headers.ServerVersion
	app.ArgsUsage = " "
	app.Flags = []cli.Flag{
		&cli.StringFlag{
			Name:    "root",
			Aliases: []string{"r"},
			Value:   ".",
			Usage:   "root directory of execution environment",
			EnvVars: []string{config.EnvKeyRoot},
		},
		&cli.StringFlag{
			Name:    "config",
			Aliases: []string{"c"},
			Value:   "config",
			Usage:   "config dir path relative to root",
			EnvVars: []string{config.EnvKeyConfigDir},
		},
		&cli.StringFlag{
			Name:    "acl-config",
			Aliases: []string{"ac"},
			Value:   "acl.yaml",
			Usage:   "acl config path relative to root",
			EnvVars: []string{"TEMPORAL_ACL_CONFIG_PATH"},
		},
		&cli.StringFlag{
			Name:    "env",
			Aliases: []string{"e"},
			Value:   "development",
			Usage:   "runtime environment",
			EnvVars: []string{config.EnvKeyEnvironment},
		},
		&cli.StringFlag{
			Name:    "zone",
			Aliases: []string{"az"},
			Usage:   "availability zone",
			EnvVars: []string{config.EnvKeyAvailabilityZone, config.EnvKeyAvailabilityZoneTypo},
		},
	}

	app.Commands = []*cli.Command{
		{
			Name:      "start",
			Usage:     "Start Temporal server",
			ArgsUsage: " ",
			Flags: []cli.Flag{
				&cli.StringSliceFlag{
					Name:    "service",
					Aliases: []string{"svc"},
					Value:   cli.NewStringSlice(temporal.Services...),
					Usage:   "service(s) to start",
				},
			},
			Action: func(c *cli.Context) error {
				env := c.String("env")
				zone := c.String("zone")
				configDir := path.Join(c.String("root"), c.String("config"))
				services := c.StringSlice("service")
				aclConfigPath := path.Join(c.String("root"), c.String("acl-config"))

				cfg, err := config.LoadConfig(env, configDir, zone)
				if err != nil {
					return cli.Exit(fmt.Sprintf("Unable to load configuration: %v", err), 1)
				}

				data, err := ioutil.ReadFile(aclConfigPath)
				if err != nil {
					return cli.Exit(fmt.Sprintf("Unable to load ACL configuration: %v", err), 1)
				}

				acl := make(map[string][]string)
				err = yaml.Unmarshal(data, &acl)
				if err != nil {
					return cli.Exit(fmt.Sprintf("Unable to load ACL configuration: %v", err), 1)
				}

				logger := tlog.NewZapLogger(tlog.BuildZapLogger(cfg.Log))
				authorizer, err := NewSwatAuthorizer(logger, acl)
				if err != nil {
					return cli.Exit(fmt.Sprintf("Unable to configure authorizer: %v", err), 1)
				}

				s := temporal.NewServer(
					temporal.ForServices(services),
					temporal.WithConfig(cfg),
					temporal.WithLogger(logger),
					temporal.InterruptOn(temporal.InterruptCh()),
					temporal.WithAuthorizer(authorizer),
					temporal.WithClaimMapper(NewSwatClaimMapper),
				)

				err = s.Start()
				if err != nil {
					return cli.Exit(fmt.Sprintf("Unable to start server. Error: %v", err), 1)
				}
				return cli.Exit("All services are stopped.", 0)
			},
		},
	}
	return app
}
