// based on vendor/go.temporal.io/server/tools/cli/factory.go

package main

import (
	"context"
	"crypto/tls"
	"github.com/urfave/cli"
	"go.temporal.io/api/workflowservice/v1"
	"go.temporal.io/sdk/client"
	"go.temporal.io/server/api/adminservice/v1"
	"go.temporal.io/server/common/log"
	"go.temporal.io/server/common/log/tag"
	tctl "go.temporal.io/server/tools/cli"
	"google.golang.org/grpc"
	"google.golang.org/grpc/credentials"
	healthpb "google.golang.org/grpc/health/grpc_health_v1"
	"google.golang.org/grpc/metadata"
	"net"
)

type clientFactory struct {
	logger     log.Logger
	oauthToken string
}

// NewClientFactory creates a new ClientFactory
func NewClientFactory(oauthToken string) tctl.ClientFactory {
	logger := log.NewCLILogger()

	return &clientFactory{
		logger:     logger,
		oauthToken: oauthToken,
	}
}

// FrontendClient builds a frontend client
func (b *clientFactory) FrontendClient(c *cli.Context) workflowservice.WorkflowServiceClient {
	connection, _ := b.createGRPCConnection(c)

	return workflowservice.NewWorkflowServiceClient(connection)
}

// AdminClient builds an admin client.
func (b *clientFactory) AdminClient(c *cli.Context) adminservice.AdminServiceClient {
	connection, _ := b.createGRPCConnection(c)
	return adminservice.NewAdminServiceClient(connection)
}

type headersProvider struct {
	oauthToken string
}

func (h *headersProvider) GetHeaders(ctx context.Context) (map[string]string, error) {
	headers := make(map[string]string)
	headers["Authorization"] = h.oauthToken
	return headers, nil
}

// SDKClient builds an SDK client.
func (b *clientFactory) SDKClient(c *cli.Context, namespace string) client.Client {
	hostPort := c.GlobalString(tctl.FlagAddress)

	tlsConfig, err := b.createTLSConfig(c)
	if err != nil {
		b.logger.Fatal("Failed to configure TLS for SDK client", tag.Error(err))
	}

	sdkClient, err := client.NewClient(client.Options{
		HostPort:  hostPort,
		Namespace: namespace,
		Logger:    log.NewSdkLogger(b.logger),
		ConnectionOptions: client.ConnectionOptions{
			DisableHealthCheck: true,
			TLS:                tlsConfig,
		},
		HeadersProvider: &headersProvider{oauthToken: b.oauthToken},
	})
	if err != nil {
		b.logger.Fatal("Failed to create SDK client", tag.Error(err))
	}

	return sdkClient
}

// HealthClient builds a health client.
func (b *clientFactory) HealthClient(c *cli.Context) healthpb.HealthClient {
	connection, _ := b.createGRPCConnection(c)
	return healthpb.NewHealthClient(connection)
}

func headersProviderInterceptor(headersProvider headersProvider) grpc.UnaryClientInterceptor {
	return func(ctx context.Context, method string, req, reply interface{}, cc *grpc.ClientConn, invoker grpc.UnaryInvoker, opts ...grpc.CallOption) error {
		headers, err := headersProvider.GetHeaders(ctx)
		if err != nil {
			return err
		}
		for k, v := range headers {
			ctx = metadata.AppendToOutgoingContext(ctx, k, v)
		}
		return invoker(ctx, method, req, reply, cc, opts...)
	}
}

func (b *clientFactory) createGRPCConnection(c *cli.Context) (*grpc.ClientConn, error) {
	hostPort := c.GlobalString(tctl.FlagAddress)
	tlsConfig, err := b.createTLSConfig(c)
	if err != nil {
		return nil, err
	}
	grpcSecurityOptions := grpc.WithInsecure()
	if tlsConfig != nil {
		grpcSecurityOptions = grpc.WithTransportCredentials(credentials.NewTLS(tlsConfig))
	}

	x := grpc.WithChainUnaryInterceptor(
		headersProviderInterceptor(headersProvider{b.oauthToken}),
	)
	connection, err := grpc.Dial(hostPort, grpcSecurityOptions, x)
	if err != nil {
		b.logger.Fatal("Failed to create connection", tag.Error(err))
		return nil, err
	}
	return connection, nil
}

func (b *clientFactory) createTLSConfig(c *cli.Context) (*tls.Config, error) {
	hostPort := c.GlobalString(tctl.FlagAddress)
	host, _, err := net.SplitHostPort(hostPort)
	if err != nil {
		return nil, err
	}
	return &tls.Config{
		MinVersion: tls.VersionTLS12,
		NextProtos: []string{
			"h2",
		},
		ServerName: host,
	}, nil
}
