package awacs

import (
	"a.yandex-team.ru/infra/temporal/activities/calendar"
	"a.yandex-team.ru/infra/temporal/workflows/common"
	"fmt"
	"google.golang.org/protobuf/types/known/timestamppb"
	"google.golang.org/protobuf/types/known/wrapperspb"
	"sort"
	"strings"
	"time"

	awacsactivities "a.yandex-team.ru/infra/temporal/activities/awacs"
	"a.yandex-team.ru/library/go/slices"

	"go.temporal.io/sdk/temporal"
	"go.temporal.io/sdk/workflow"

	awacspb "a.yandex-team.ru/infra/awacs/proto"
)

var validLocations = []string{"SAS", "MAN", "VLA", "MYT", "IVA", "UNKNOWN"}

func getBalancerLocation(balancer *awacspb.Balancer) (string, error) {
	meta := balancer.GetMeta()
	location := meta.GetLocation()
	if location == nil {
		return "", fmt.Errorf("no location specified for %s/%s", meta.NamespaceId, meta.Id)
	}

	var rv string
	if location.Type == awacspb.BalancerMeta_Location_YP_CLUSTER {
		rv = location.GetYpCluster()
		if rv == "MAN_PRE" || rv == "SAS_TEST" || rv == "TEST_SAS" {
			rv = "UNKNOWN"
		}
	} else if location.Type == awacspb.BalancerMeta_Location_GENCFG_DC {
		rv = location.GetGencfgDc()
	} else if location.Type == awacspb.BalancerMeta_Location_UNKNOWN {
		rv = "UNKNOWN"
	}
	if !slices.ContainsString(validLocations, rv) {
		return "", fmt.Errorf("failed to determine location of %s/%s", meta.NamespaceId, meta.Id)
	}
	return rv, nil
}

func groupBalancersByLocation(balancers []*awacspb.Balancer) (map[string][]string, error) {
	rv := make(map[string][]string)
	for _, balancer := range balancers {
		location, err := getBalancerLocation(balancer)
		if err != nil {
			return nil, err
		}
		rv[location] = append(rv[location], balancer.GetMeta().Id)
	}
	return rv, nil
}

func extendCertificateVisibility(ctx, commonCtx, pollingCtx workflow.Context, namespaceID, certID, certRevID string,
	locations []string, balancerIdsByLocation map[string][]string) error {
	var a *awacsactivities.Activities
	var err error
	sort.Strings(locations)
	for _, location := range locations {
		err = workflow.ExecuteActivity(commonCtx, a.ExtendCertificateDiscoverability,
			namespaceID, certID, location).Get(ctx, nil)
		if err != nil {
			return err
		}

		err = workflow.ExecuteActivity(pollingCtx, a.WaitUntilCertificateIsActiveInBalancers,
			namespaceID, certID, certRevID, balancerIdsByLocation[location]).Get(ctx, nil)
		if err != nil {
			return err
		}
	}

	err = workflow.ExecuteActivity(commonCtx, a.MakeCertificateDiscoverableByDefault, namespaceID, certID).Get(ctx, nil)
	if err != nil {
		return err
	}

	return nil
}

func ContinueRenewedCertificateDeployment(ctx workflow.Context, namespaceID, certID string) error {
	commonCtx := workflow.WithActivityOptions(ctx, workflow.ActivityOptions{
		StartToCloseTimeout: time.Minute * 10,
		HeartbeatTimeout:    time.Minute * 2,
		RetryPolicy: &temporal.RetryPolicy{
			InitialInterval:        time.Second * 30,
			BackoffCoefficient:     1.1,
			MaximumInterval:        time.Minute * 5,
			MaximumAttempts:        30,
			NonRetryableErrorTypes: []string{"PermanentError"},
		},
	})

	pollingCtx := workflow.WithActivityOptions(ctx, workflow.ActivityOptions{
		StartToCloseTimeout: time.Hour * 24,
		HeartbeatTimeout:    time.Minute * 2,
		RetryPolicy: &temporal.RetryPolicy{
			InitialInterval:        time.Minute,
			BackoffCoefficient:     10,
			MaximumInterval:        time.Minute,
			MaximumAttempts:        5,
			NonRetryableErrorTypes: []string{"PermanentError"},
		},
	})

	var a *awacsactivities.Activities

	var balancers []*awacspb.Balancer
	err := workflow.ExecuteActivity(commonCtx, a.ListBalancersByNamespaceID, namespaceID).Get(ctx, &balancers)
	if err != nil {
		return err
	}
	balancerIdsByLocation, err := groupBalancersByLocation(balancers)
	if err != nil {
		return err
	}
	locations := make([]string, 0)
	for location := range balancerIdsByLocation {
		locations = append(locations, location)
	}

	var cert *awacspb.Certificate
	err = workflow.ExecuteActivity(commonCtx, a.GetCertificate, namespaceID, certID).Get(ctx, &cert)
	if err != nil {
		return err
	}

	if cert.Meta.GetDiscoverability() == nil || cert.Meta.Discoverability.Default.Value {
		return nil
	}

	certRevID := cert.Meta.Version

	return extendCertificateVisibility(ctx, commonCtx, pollingCtx,
		namespaceID, certID, certRevID, locations, balancerIdsByLocation)
}

func DeployCertificateRenewal(ctx workflow.Context, namespaceID, certID string) error {
	commonCtx := workflow.WithActivityOptions(ctx, workflow.ActivityOptions{
		StartToCloseTimeout: time.Minute * 10,
		RetryPolicy: &temporal.RetryPolicy{
			InitialInterval:        time.Second * 30,
			BackoffCoefficient:     1.1,
			MaximumInterval:        time.Minute * 5,
			MaximumAttempts:        30,
			NonRetryableErrorTypes: []string{"PermanentError"},
		},
	})

	pollingCtx := workflow.WithActivityOptions(ctx, workflow.ActivityOptions{
		StartToCloseTimeout: time.Hour * 24,
		HeartbeatTimeout:    time.Minute * 2,
		RetryPolicy: &temporal.RetryPolicy{
			InitialInterval:        time.Minute,
			BackoffCoefficient:     10,
			MaximumInterval:        time.Minute,
			MaximumAttempts:        5,
			NonRetryableErrorTypes: []string{"PermanentError"},
		},
	})

	var a *awacsactivities.Activities

	var balancers []*awacspb.Balancer
	err := workflow.ExecuteActivity(commonCtx, a.ListBalancersByNamespaceID, namespaceID).Get(ctx, &balancers)
	if err != nil {
		return err
	}
	var balancerIds []string
	for _, balancer := range balancers {
		balancerIds = append(balancerIds, balancer.GetMeta().Id)
	}
	balancerIdsByLocation, err := groupBalancersByLocation(balancers)
	if err != nil {
		return err
	}
	locations := make([]string, 0)
	for location := range balancerIdsByLocation {
		locations = append(locations, location)
	}

	err = workflow.ExecuteActivity(commonCtx, a.WaitUntilNamespaceIsUnpausedAndSettled, namespaceID).Get(ctx, nil)
	if err != nil {
		return err
	}

	var cert *awacspb.Certificate
	err = workflow.ExecuteActivity(commonCtx, a.GetCertificate, namespaceID, certID).Get(ctx, &cert)
	if err != nil {
		return err
	}

	if len(cert.GetStatuses()) != 1 {
		return fmt.Errorf("certificate %s/%s has more than one present revision", namespaceID, certID)
	}

	certBalancerIds := make([]string, 0)
	for flatBalancerID, activeStatus := range cert.GetStatuses()[0].GetActive() {
		if activeStatus.Status == "True" {
			certBalancerIds = append(certBalancerIds, strings.Split(flatBalancerID, ":")[1])
		}
	}
	if !slices.EqualAnyOrderStrings(balancerIds, certBalancerIds) {
		return fmt.Errorf("certificate %s/%s is not active in all namespace balancers", namespaceID, certID)
	}

	var certRenewal *awacspb.CertificateRenewal
	err = workflow.ExecuteActivity(commonCtx, a.GetCertificateRenewal, namespaceID, certID).Get(ctx, &certRenewal)
	if err != nil {
		return err
	}

	targetDiscoverability := &awacspb.DiscoverabilityCondition{
		Default: &awacspb.BoolCondition{Value: false},
	}
	err = workflow.ExecuteActivity(commonCtx, a.UnpauseCertificateRenewal, namespaceID, certID, targetDiscoverability).Get(ctx, nil)
	if err != nil {
		return err
	}

	var certRevID string
	err = workflow.ExecuteActivity(pollingCtx, a.WaitUntilCertificateIsRenewed, namespaceID, certID).Get(ctx, &certRevID)
	if err != nil {
		return err
	}

	return extendCertificateVisibility(ctx, commonCtx, pollingCtx,
		namespaceID, certID, certRevID, locations, balancerIdsByLocation)
}

var a *awacsactivities.Activities

func isOkToContinue(ctx workflow.Context) (bool, error) {
	var isWorkingDay bool
	err := workflow.ExecuteActivity(ctx, calendar.IsTodayWorkingDay).Get(ctx, &isWorkingDay)
	if err != nil {
		return false, fmt.Errorf("failed to check whether it's a working day: %w", err)
	}
	if !isWorkingDay {
		return false, fmt.Errorf("it's not a working day, not proceeding")
	}

	var isWorkingHours bool
	err = workflow.ExecuteActivity(ctx, calendar.IsNowWorkingHours).Get(ctx, &isWorkingHours)
	if err != nil {
		return false, fmt.Errorf("failed to check whether it's working hours: %w", err)
	}
	if !isWorkingHours {
		return false, fmt.Errorf("it's not working hours, not proceeding")
	}
	return true, nil
}

func ScheduleCertificateRenewalDeployments(ctx workflow.Context) error {
	ctx = workflow.WithActivityOptions(ctx, workflow.ActivityOptions{
		StartToCloseTimeout: time.Minute * 10,
		RetryPolicy: &temporal.RetryPolicy{
			InitialInterval:        time.Second * 30,
			BackoffCoefficient:     1.1,
			MaximumInterval:        time.Minute * 5,
			MaximumAttempts:        30,
			NonRetryableErrorTypes: []string{"PermanentError"},
		},
	})

	if ok, err := isOkToContinue(ctx); !ok {
		return err
	}

	var renewals []*awacspb.CertificateRenewal
	err := workflow.ExecuteActivity(ctx, a.ListCertificateRenewals, &awacspb.ListCertificateRenewalsRequest_Query{
		Incomplete: &wrapperspb.BoolValue{Value: false},
		Paused:     &wrapperspb.BoolValue{Value: true},
		ValidityNotBefore: &awacspb.TimestampRangeQuery{
			Lte: timestamppb.New(workflow.Now(ctx).Add(-24 * time.Hour)),
		},
	}).Get(ctx, &renewals)
	if err != nil {
		return err
	}

	pool := common.NewChildWorkflowPool(ctx, 3)

	for _, renewal := range renewals {
		if ok, err := isOkToContinue(ctx); !ok {
			return fmt.Errorf("%w: %s", err, pool.Wait())
		}
		namespaceID := renewal.Meta.NamespaceId
		certID := renewal.Meta.Id
		cwo := workflow.ChildWorkflowOptions{
			WorkflowID: fmt.Sprintf("deploy-certificate-renewal(%s/%s)", namespaceID, certID),
		}
		_, err = pool.ExecuteChildWorkflow(cwo, DeployCertificateRenewal, namespaceID, certID)
		if err != nil {
			return err
		}
	}
	return pool.Wait()
}
